"use client";

import { useState, useEffect } from "react";
import { X, Loader2, Save } from "lucide-react";

interface DriverModalProps {
  isOpen: boolean;
  onClose: () => void;
  onSave: (data: any) => Promise<void>;
  driver?: any; // If provided, we are in Edit mode
}

export default function DriverModal({ isOpen, onClose, onSave, driver }: DriverModalProps) {
  const [loading, setLoading] = useState(false);
  const [formData, setFormData] = useState<any>({
    name: "",
    email: "",
    mobile_number: "",
    password: "",
    riding_mode: "courier",
    gender: "male",
    ride_serv_pref: "package",
    vehicle_number: "",
    hot_food_bag: "false",
    is_verified: 0,
    current_latitude: "",
    current_longitude: "",
    status: "pending",
  });

  useEffect(() => {
    if (driver) {
      setFormData({
        name: driver.user?.name || "",
        email: driver.user?.email || "",
        mobile_number: driver.user?.mobile_number || driver.user?.mobile || "",
        password: "", // Password usually not returned
        riding_mode: driver.meta?.riding_mode || "courier",
        gender: driver.meta?.gender || "male",
        ride_serv_pref: driver.meta?.ride_serv_pref || "package",
        vehicle_number: driver.meta?.vehicle_number || "",
        hot_food_bag: String(driver.meta?.hot_food_bag || "false"),
        is_verified: driver.is_verified || 0,
        current_latitude: driver.current_latitude || "",
        current_longitude: driver.current_longitude || "",
        status: driver.status || "pending",
      });
    } else {
      setFormData({
        name: "",
        email: "",
        mobile_number: "",
        password: "",
        riding_mode: "courier",
        gender: "male",
        ride_serv_pref: "package",
        vehicle_number: "",
        hot_food_bag: "false",
        is_verified: 0,
        current_latitude: "",
        current_longitude: "",
        status: "pending",
      });
    }
  }, [driver, isOpen]);

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value, type } = e.target as HTMLInputElement;
    setFormData((prev: any) => ({
      ...prev,
      [name]: type === "checkbox" ? ((e.target as HTMLInputElement).checked ? 1 : 0) : value,
    }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    try {
      // Prepare payload
      const payload: any = {
        is_verified: formData.is_verified,
        status: formData.status,
        meta: JSON.stringify({
          riding_mode: formData.riding_mode,
          gender: formData.gender,
          ride_serv_pref: formData.ride_serv_pref,
          vehicle_number: formData.vehicle_number,
          hot_food_bag: formData.hot_food_bag,
        }),
      };

      if (!driver) {
        // Only for create
        payload.name = formData.name;
        payload.email = formData.email;
        payload.mobile_number = formData.mobile_number;
        payload.password = formData.password;
      }

      // Latitude and Longitude are top level in the model but can be null
      if (formData.current_latitude) payload.current_latitude = formData.current_latitude;
      if (formData.current_longitude) payload.current_longitude = formData.current_longitude;

      await onSave(payload);
      onClose();
    } catch (error) {
      console.error("Error saving driver:", error);
    } finally {
      setLoading(false);
    }
  };

  if (!isOpen) return null;

  return (
    <div className="modal-overlay position-fixed top-0 start-0 w-100 h-100 d-flex align-items-center justify-content-center" style={{ zIndex: 1050, backgroundColor: 'rgba(0,0,0,0.5)', backdropFilter: 'blur(4px)' }}>
      <div className="card-custom m-3 shadow-lg border-0 overflow-hidden" style={{ width: '100%', maxWidth: '700px', maxHeight: '90vh', display: 'flex', flexDirection: 'column' }}>
        {/* Header */}
        <div className="p-4 border-bottom d-flex justify-content-between align-items-center bg-white">
          <h5 className="fw-bold mb-0">{driver ? "Edit Driver" : "Create New Driver"}</h5>
          <button onClick={onClose} className="btn btn-light rounded-circle p-2 border-0">
            <X size={20} />
          </button>
        </div>

        {/* Body */}
        <form onSubmit={handleSubmit} className="p-4 overflow-auto bg-light">
          <div className="row g-3">
            {/* Basic Info - Only for Create */}
            {!driver && (
              <>
                <div className="col-md-6">
                  <label className="form-label small fw-bold">Name</label>
                  <input type="text" name="name" value={formData.name} onChange={handleChange} className="form-control" required placeholder="Full Name" />
                </div>
                <div className="col-md-6">
                  <label className="form-label small fw-bold">Email</label>
                  <input type="email" name="email" value={formData.email} onChange={handleChange} className="form-control" required placeholder="email@example.com" />
                </div>
                <div className="col-md-6">
                  <label className="form-label small fw-bold">Mobile Number</label>
                  <input type="text" name="mobile_number" value={formData.mobile_number} onChange={handleChange} className="form-control" required placeholder="+20..." />
                </div>
                <div className="col-md-6">
                  <label className="form-label small fw-bold">Password</label>
                  <input type="password" name="password" value={formData.password} onChange={handleChange} className="form-control" required={!driver} placeholder="••••••••" />
                </div>
              </>
            )}

            {/* Meta Info */}
            <div className="col-md-4">
              <label className="form-label small fw-bold">Riding Mode</label>
              <select name="riding_mode" value={formData.riding_mode} onChange={handleChange} className="form-select">
                <option value="courier">Courier</option>
                <option value="taxi">Taxi</option>
                <option value="delivery">Delivery</option>
              </select>
            </div>
            <div className="col-md-4">
              <label className="form-label small fw-bold">Gender</label>
              <select name="gender" value={formData.gender} onChange={handleChange} className="form-select">
                <option value="male">Male</option>
                <option value="female">Female</option>
              </select>
            </div>
            <div className="col-md-4">
              <label className="form-label small fw-bold">Ride Service Preference</label>
              <select name="ride_serv_pref" value={formData.ride_serv_pref} onChange={handleChange} className="form-select">
                <option value="package">Package</option>
                <option value="ride">Ride</option>
                <option value="both">Both</option>
              </select>
            </div>

            <div className="col-md-6">
              <label className="form-label small fw-bold">Vehicle Number</label>
              <input type="text" name="vehicle_number" value={formData.vehicle_number} onChange={handleChange} className="form-control" placeholder="ABC-123" />
            </div>
            <div className="col-md-6">
              <label className="form-label small fw-bold">Hot Food Bag</label>
              <select name="hot_food_bag" value={formData.hot_food_bag} onChange={handleChange} className="form-select">
                <option value="true">Yes</option>
                <option value="false">No</option>
              </select>
            </div>

            {/* Location */}
            <div className="col-md-6">
              <label className="form-label small fw-bold">Latitude</label>
              <input type="text" name="current_latitude" value={formData.current_latitude} onChange={handleChange} className="form-control" placeholder="30.0..." />
            </div>
            <div className="col-md-6">
              <label className="form-label small fw-bold">Longitude</label>
              <input type="text" name="current_longitude" value={formData.current_longitude} onChange={handleChange} className="form-control" placeholder="31.2..." />
            </div>

            {/* Status & Verification */}
            <div className="col-md-6">
              <label className="form-label small fw-bold">Account Status</label>
              <select name="status" value={formData.status} onChange={handleChange} className="form-select">
                <option value="pending">Pending</option>
                <option value="approved">Approved</option>
                <option value="rejected">Rejected</option>
                <option value="suspended">Suspended</option>
              </select>
            </div>

            <div className="col-md-6 d-flex align-items-end pb-1">
              <div className="form-check form-switch mb-2">
                <input className="form-check-input" type="checkbox" name="is_verified" id="isVerifiedSwitch" checked={formData.is_verified === 1} onChange={(e) => setFormData((p: any) => ({ ...p, is_verified: e.target.checked ? 1 : 0 }))} />
                <label className="form-check-label small fw-bold ms-2" htmlFor="isVerifiedSwitch">Is Verified?</label>
              </div>
            </div>
          </div>

          {/* Actions */}
          <div className="d-flex justify-content-end gap-2 mt-4 pt-3 border-top">
            <button type="button" onClick={onClose} className="btn btn-outline-secondary px-4 py-2 rounded-3 border-0">
              Cancel
            </button>
            <button type="submit" disabled={loading} className="btn btn-primary px-4 py-2 rounded-3 fw-bold border-0 d-flex align-items-center gap-2" style={{ background: 'linear-gradient(135deg, #f97316 0%, #ea580c 100%)' }}>
              {loading ? <Loader2 size={18} className="spin-animation" /> : <Save size={18} />}
              {driver ? "Update Driver" : "Create Driver"}
            </button>
          </div>
        </form>
      </div>
    </div>
  );
}
