"use client";

import { useState } from "react";
import Link from "next/link";
import { usePathname, useRouter } from "next/navigation";
import { useAuth } from "@/context/AuthContext";
import { 
  LayoutDashboard, 
  Users, 
  Store, 
  ShoppingBag, 
  Settings, 
  LogOut, 
  Car, 
  BarChart3, 
  Wallet,
  Package,
  Image,
  ChevronDown,
  ChevronRight,
  Circle,
  CreditCard,
  DollarSign
} from "lucide-react";

interface MenuItem {
  label: string;
  icon?: any;
  href?: string;
  items?: { label: string; href: string }[];
}

const MENU_ITEMS: MenuItem[] = [
  { 
    label: "Dashboard", 
    icon: LayoutDashboard, 
    href: "/admin" 
  },
  { 
    label: "Analytics", 
    icon: BarChart3, 
    href: "/admin/analytics" 
  },
  {
    label: "User Management",
    icon: Users,
    items: [
      { label: "All Users", href: "/admin/users" },
      { label: "Create User", href: "/admin/users/create" },
    ]
  },
  {
    label: "Store Management",
    icon: Store,
    items: [
      { label: "All Stores", href: "/admin/stores" },
      { label: "Create Store", href: "/admin/stores/create" },
      { label: "Providers", href: "/admin/providers" },
    ]
  },
  {
    label: "Product Management",
    icon: Package,
    items: [
      { label: "All Products", href: "/admin/products" },
      { label: "Create Product", href: "/admin/products/create" },
      { label: "Categories", href: "/admin/categories" },
    ]
  },
  {
    label: "Order Management",
    icon: ShoppingBag,
    items: [
      { label: "All Orders", href: "/admin/orders" },
      { label: "Appointments", href: "/admin/appointments" },
      { label: "Active Orders", href: "/admin/orders/active" },
    ]
  },
  {
    label: "Ride Hailing",
    icon: Car,
    items: [
      { label: "All Rides", href: "/admin/rides" },
      { label: "Drivers", href: "/admin/drivers" },
      { label: "Vehicle Types", href: "/admin/vehicle-types" },
      { label: "Location Charges", href: "/admin/location-charges" },
    ]
  },
  {
    label: "Finance",
    icon: Wallet,
    href: "/admin/finance"
  },
  {
    label: "Content Management",
    icon: Image,
    items: [
      { label: "All Banners", href: "/admin/banners" },
      { label: "Create Banner", href: "/admin/banners/create" },
      { label: "Coupons", href: "/admin/coupons" },
    ]
  },
  {
    label: "System",
    icon: Settings,
    href: "/admin/settings"
  }
];

export default function AdminSidebar({
  width = 280,
  collapsed = false
}: {
  width?: number;
  collapsed?: boolean;
}) {
  const pathname = usePathname();
  const [expanded, setExpanded] = useState<string[]>([]);
  const { logout } = useAuth();
  const router = useRouter();

  const handleLogout = () => {
    logout();
    router.push("/login");
  };

  const toggleSection = (label: string) => {
    if (collapsed) return; // Don't expand in collapsed mode
    setExpanded(prev =>
      prev.includes(label)
        ? prev.filter(l => l !== label)
        : [...prev, label]
    );
  };

  return (
    <aside className="position-fixed top-0 start-0 border-end transition-all duration-300 bg-sidebar" style={{ 
      width: `${width}px`,
      zIndex: 1050,
      borderColor: 'var(--border-color)',
      transition: 'width 0.3s cubic-bezier(0.4, 0, 0.2, 1)'
    }}>
      {/* Brand */}
      <div className={`d-flex align-items-center ${collapsed ? 'justify-content-center px-0' : 'justify-content-start gap-3 px-4'} py-4 border-bottom mb-2 transition-all`} style={{ borderColor: 'var(--border-color)' }}>
        <div className="rounded-3 d-flex align-items-center justify-content-center fw-bold text-white shadow-sm flex-shrink-0" style={{
          width: '40px',
          height: '40px',
          background: 'linear-gradient(135deg, #f97316 0%, #ea580c 100%)',
        }}>
          D
        </div>
        {!collapsed && (
          <div className="overflow-hidden whitespace-nowrap">
            <h6 className="mb-0 fw-bold tracking-tight" style={{ color: 'var(--text-color)' }}>Deligo Admin</h6>
            <small className="text-muted-custom" style={{ fontSize: '10px', textTransform: 'uppercase', letterSpacing: '0.5px' }}>Control Panel</small>
          </div>
        )}
      </div>

      {/* Navigation */}
      <div className={`flex-1 overflow-y-auto ${collapsed ? 'px-2' : 'px-3'} pb-4 no-scrollbar`}>
        <ul className="nav flex-column gap-1">
          {MENU_ITEMS.map((item, idx) => {
            const isActive = item.href === pathname;
            const isExpanded = expanded.includes(item.label);
            const hasChildren = item.items && item.items.length > 0;
            const isChildActive = hasChildren && item.items?.some(sub => sub.href === pathname);

            return (
              <li key={idx} className="nav-item">
                {hasChildren ? (
                  <>
                    <button 
                      onClick={() => toggleSection(item.label)}
                      className={`nav-link w-100 d-flex align-items-center ${collapsed ? 'justify-content-center' : 'justify-content-between'} text-start py-2.5 rounded-3 mb-1`}
                      style={{ 
                        border: 'none',
                        background: isChildActive || (isExpanded && !collapsed) ? 'var(--hover-bg)' : 'transparent',
                        cursor: 'pointer',
                        paddingLeft: collapsed ? '0' : '1rem',
                        paddingRight: collapsed ? '0' : '1rem',
                        color: isChildActive ? 'var(--text-color)' : 'var(--muted-text)'
                      }}
                      title={collapsed ? item.label : undefined}
                      onMouseEnter={(e) => e.currentTarget.style.color = 'var(--text-color)'}
                      onMouseLeave={(e) => e.currentTarget.style.color = (isChildActive || isExpanded) ? 'var(--text-color)' : 'var(--muted-text)'}
                    >
                      <div className={`d-flex align-items-center ${collapsed ? 'justify-content-center' : 'gap-3'}`}>
                        {item.icon && <item.icon size={18} />}
                        {!collapsed && <span className="small fw-medium">{item.label}</span>}
                      </div>
                      {!collapsed && (isExpanded ? <ChevronDown size={14} /> : <ChevronRight size={14} />)}
                    </button>
                    
                    {/* Submenu Dropdown */}
                    {!collapsed && (
                      <div className={`overflow-hidden transition-all ${isExpanded ? 'd-block' : 'd-none'}`}>
                        <ul className="nav flex-column gap-1 ps-3 ms-2 mb-2 border-start border-opacity-25" style={{ borderColor: 'var(--border-color)' }}>
                          {item.items?.map((subItem, subIdx) => {
                            const isSubActive = pathname === subItem.href;
                            return (
                              <li key={subIdx} className="nav-item">
                                <Link 
                                  href={subItem.href}
                                  className={`nav-link d-flex align-items-center gap-2 py-2 px-3 rounded-2`}
                                  style={{ 
                                    fontSize: '0.85rem',
                                    color: isSubActive ? 'var(--text-color)' : 'var(--muted-text)' 
                                  }}
                                  onMouseEnter={(e) => e.currentTarget.style.color = 'var(--text-color)'}
                                  onMouseLeave={(e) => e.currentTarget.style.color = isSubActive ? 'var(--text-color)' : 'var(--muted-text)'}
                                >
                                  <span className={`rounded-circle ${isSubActive ? 'bg-primary' : 'bg-secondary'}`} style={{ width: '4px', height: '4px', backgroundColor: isSubActive ? 'var(--primary)' : 'var(--muted-text)' }}></span>
                                  {subItem.label}
                                </Link>
                              </li>
                            );
                          })}
                        </ul>
                      </div>
                    )}
                  </>
                ) : (
                  <Link 
                    href={item.href || "#"}
                    className={`nav-link d-flex align-items-center ${collapsed ? 'justify-content-center' : 'gap-3 px-3'} py-2.5 rounded-3 mb-1`}
                    style={{
                      background: isActive ? 'linear-gradient(135deg, #f97316 0%, #ea580c 100%)' : 'transparent',
                      color: isActive ? '#ffffff' : 'var(--muted-text)'
                    }}
                    title={collapsed ? item.label : undefined}
                    onMouseEnter={(e) => {
                      if (!isActive) e.currentTarget.style.color = 'var(--text-color)';
                    }}
                    onMouseLeave={(e) => {
                      if (!isActive) e.currentTarget.style.color = 'var(--muted-text)';
                    }}
                  >
                    {item.icon && <item.icon size={18} />}
                    {!collapsed && <span className="small fw-medium">{item.label}</span>}
                  </Link>
                )}
              </li>
            );
          })}
          {/* Transactions & Payouts */}
          <li className="nav-item">
            <Link 
              href="/admin/transactions"
              className={`nav-link d-flex align-items-center ${collapsed ? 'justify-content-center' : 'gap-3 px-3'} py-2.5 rounded-3 mb-1`}
              style={{
                background: pathname === "/admin/transactions" ? 'linear-gradient(135deg, #f97316 0%, #ea580c 100%)' : 'transparent',
                color: pathname === "/admin/transactions" ? '#ffffff' : 'var(--muted-text)'
              }}
              title={collapsed ? "Transactions" : undefined}
              onMouseEnter={(e) => {
                if (pathname !== "/admin/transactions") e.currentTarget.style.color = 'var(--text-color)';
              }}
              onMouseLeave={(e) => {
                if (pathname !== "/admin/transactions") e.currentTarget.style.color = 'var(--muted-text)';
              }}
            >
              <CreditCard size={18} />
              {!collapsed && <span className="small fw-medium">Transactions</span>}
            </Link>
          </li>
          <li className="nav-item">
            <Link 
              href="/admin/payouts"
              className={`nav-link d-flex align-items-center ${collapsed ? 'justify-content-center' : 'gap-3 px-3'} py-2.5 rounded-3 mb-1`}
              style={{
                background: pathname === "/admin/payouts" ? 'linear-gradient(135deg, #f97316 0%, #ea580c 100%)' : 'transparent',
                color: pathname === "/admin/payouts" ? '#ffffff' : 'var(--muted-text)'
              }}
              title={collapsed ? "Payouts" : undefined}
              onMouseEnter={(e) => {
                if (pathname !== "/admin/payouts") e.currentTarget.style.color = 'var(--text-color)';
              }}
              onMouseLeave={(e) => {
                if (pathname !== "/admin/payouts") e.currentTarget.style.color = 'var(--muted-text)';
              }}
            >
              <DollarSign size={18} />
              {!collapsed && <span className="small fw-medium">Payouts</span>}
            </Link>
          </li>
        </ul>
      </div>

      {/* Footer / Logout */}
      <div className={`p-3 border-top mt-auto transition-all`} style={{ borderColor: 'var(--border-color)' }}>
        <button 
          onClick={handleLogout}
          className={`btn w-100 d-flex align-items-center ${collapsed ? 'justify-content-center' : 'justify-content-center gap-2'} text-danger text-opacity-75 hover:bg-danger hover:bg-opacity-10 rounded-3 py-2 transition-all`}
        >
          <LogOut size={16} />
          {!collapsed && <span className="small fw-bold">Sign Out</span>}
        </button>
      </div>
    </aside>
  );
}
