"use client";

import { useState } from "react";
import { useAuth } from "@/context/AuthContext";
import { AuthService } from "@/services/api";
import { useRouter } from "next/navigation";
import { Lock, Mail, ArrowRight, AlertCircle, Loader2 } from "lucide-react";

export default function LoginPage() {
  const [email, setEmail] = useState("");
  const [password, setPassword] = useState("");
  const [error, setError] = useState("");
  const [loading, setLoading] = useState(false);
  const { login } = useAuth();
  const router = useRouter();

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    setError("");

    try {
      const response = await AuthService.login({ email, password, role: "admin" });
      
      // Adjust based on actual API response structure
      // Common patterns: response.data.token or response.data.data.token
      const token = response.data.token || response.data.data?.token || response.data.access_token;
      const user = response.data.user || response.data.data?.user;

      if (token && user) {
        localStorage.setItem("user", JSON.stringify(user));
        await login(token, user);
        router.push("/admin");
      } else {
        throw new Error("Invalid response from server");
      }
    } catch (err: any) {
      console.error(err);
      setError(err.response?.data?.message || err.message || "Failed to login");
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-vh-100 d-flex align-items-center justify-content-center" style={{ backgroundColor: 'var(--bg-color)' }}>
      <div className="card-custom p-5 w-100" style={{ maxWidth: "420px" }}>
        
        {/* Header */}
        <div className="text-center mb-5">
          <div className="d-inline-flex align-items-center justify-content-center mb-4 rounded-circle" 
               style={{ 
                 width: '64px', 
                 height: '64px', 
                 background: 'linear-gradient(135deg, #f97316 0%, #ea580c 100%)',
                 boxShadow: '0 10px 25px -5px rgba(234, 88, 12, 0.3)'
               }}>
            <Lock size={28} className="text-white" />
          </div>
          <h2 className="fw-bold mb-2" style={{ color: 'var(--text-color)' }}>Welcome Back</h2>
          <p className="text-muted-custom">Sign in to access your admin dashboard</p>
        </div>

        {/* Error Message */}
        {error && (
          <div className="alert alert-danger d-flex align-items-center gap-2 mb-4 p-3 border-0 rounded-3" role="alert" style={{ fontSize: '0.9rem' }}>
            <AlertCircle size={18} />
            <span>{error}</span>
          </div>
        )}

        {/* Login Form */}
        <form onSubmit={handleSubmit}>
          <div className="mb-4">
            <label className="form-label small fw-bold text-muted-custom mb-2">Email Address</label>
            <div className="position-relative">
              <Mail size={18} className="position-absolute top-50 start-0 translate-middle-y ms-3 text-muted-custom" />
              <input 
                type="email" 
                className="form-control ps-5 py-2.5" 
                placeholder="admin@example.com"
                value={email}
                onChange={(e) => setEmail(e.target.value)}
                required
                style={{ 
                  backgroundColor: 'var(--bg-color)', 
                  borderColor: 'var(--border-color)',
                  color: 'var(--text-color)',
                  fontSize: '0.95rem'
                }}
              />
            </div>
          </div>

          <div className="mb-5">
             <div className="d-flex justify-content-between align-items-center mb-2">
                <label className="form-label small fw-bold text-muted-custom mb-0">Password</label>
                <a href="#" className="small fw-semibold text-primary-custom text-decoration-none">Forgot password?</a>
             </div>
            <div className="position-relative">
              <Lock size={18} className="position-absolute top-50 start-0 translate-middle-y ms-3 text-muted-custom" />
              <input 
                type="password" 
                className="form-control ps-5 py-2.5" 
                placeholder="••••••••"
                value={password}
                onChange={(e) => setPassword(e.target.value)}
                required
                style={{ 
                  backgroundColor: 'var(--bg-color)', 
                  borderColor: 'var(--border-color)',
                  color: 'var(--text-color)',
                   fontSize: '0.95rem'
                }}
              />
            </div>
          </div>

          <button 
            type="submit" 
            className="btn btn-primary w-100 py-2.5 fw-bold d-flex align-items-center justify-content-center gap-2 shadow-sm scale-on-hover"
            style={{ 
              background: 'linear-gradient(135deg, #f97316 0%, #ea580c 100%)', 
              border: 'none',
              borderRadius: '10px'
            }}
            disabled={loading}
          >
            {loading ? <Loader2 size={20} className="animate-spin" /> : <>Sign In <ArrowRight size={18} /></>}
          </button>
        </form>

      </div>
    </div>
  );
}
