"use client";

import PageHeader from "@/components/admin/PageHeader";
import ActionMenu from "@/components/admin/ActionMenu";
import DeleteModal from "@/components/admin/DeleteModal";
import { Search, Car, RefreshCw, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight } from "lucide-react";
import { useState, useEffect } from "react";
import { VehicleTypeService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";

export default function AdminVehicleTypesPage() {
  const router = useRouter();
  const [vehicleTypes, setVehicleTypes] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [deleteModalOpen, setDeleteModalOpen] = useState(false);
  const [selectedItem, setSelectedItem] = useState<any>(null);
  const [page, setPage] = useState(1);
  const [pagination, setPagination] = useState<any>(null);

  useEffect(() => {
    fetchVehicleTypes(page);
  }, [page]);

  const fetchVehicleTypes = async (page = 1) => {
    setLoading(true);
    try {
      const params: any = { page };
      if (searchTerm) {
        params.search = searchTerm;
      }
      const response = await VehicleTypeService.getAll(params);
      const resData = response.data;
      setVehicleTypes(resData.data || []);
      setPagination(resData.meta || resData);
    } catch (error) {
      console.error("Failed to fetch vehicle types:", error);
      toast.error("Failed to load vehicle types");
    } finally {
      setLoading(false);
    }
  };

  const handleDeleteClick = (item: any) => {
    setSelectedItem(item);
    setDeleteModalOpen(true);
  };

  const confirmDelete = async () => {
    if (!selectedItem) return;
    try {
      await VehicleTypeService.delete(selectedItem.id);
      toast.success("Vehicle type deleted successfully");
      setDeleteModalOpen(false);
      setSelectedItem(null);
      fetchVehicleTypes(page);
    } catch (error) {
      console.error("Failed to delete vehicle type:", error);
      toast.error("Failed to delete vehicle type");
    }
  };

  const getLocalizedValue = (value: any) => {
    if (!value) return "—";
    if (typeof value === 'string') {
      try {
        const parsed = JSON.parse(value);
        if (typeof parsed === 'object') return parsed.en || parsed.ar || Object.values(parsed)[0] || "—";
      } catch (e) {
        return value;
      }
    }
    if (typeof value === 'object') {
      return value.en || value.ar || Object.values(value)[0] || "—";
    }
    return String(value);
  };

  return (
    <div className="py-2">
      <PageHeader 
        title="Vehicle Types" 
        description="Manage vehicle types and their pricing configurations."
        actionLabel="Add New Type"
        actionLink="/admin/vehicle-types/create"
      />

      <div className="card-custom overflow-hidden">
        {/* Toolbar */}
        <div className="p-4 border-bottom d-flex flex-column flex-md-row gap-3 justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
          <div className="d-flex p-1 rounded-3" style={{ backgroundColor: 'var(--bg-color)', border: '1px solid var(--border-color)' }}>
             <button
               className="px-3 py-1.5 rounded-2 text-capitalize fw-bold border-0 transition-all text-nowrap shadow-sm"
               style={{
                 fontSize: '0.85rem',
                 backgroundColor: 'var(--text-color)',
                 color: 'var(--bg-color)',
               }}
             >
               All Types
             </button>
          </div>

          <div className="d-flex gap-2 w-100 w-md-auto">
             <div className="position-relative flex-grow-1" style={{ width: '280px' }}>
                <Search className="position-absolute top-50 translate-middle-y ms-3" size={16} style={{ color: 'var(--muted-text)' }} />
                <input 
                  type="text" 
                  placeholder="Search types..." 
                  value={searchTerm}
                  onChange={(e) => { setSearchTerm(e.target.value); setPage(1); }}
                  onKeyDown={(e) => { if (e.key === 'Enter') fetchVehicleTypes(1); }}
                  className="form-control ps-5 py-2"
                  style={{ 
                    borderRadius: '12px',
                    backgroundColor: 'var(--bg-color)',
                    borderColor: 'var(--border-color)',
                    color: 'var(--text-color)',
                    fontSize: '0.9rem'
                  }}
                />
            </div>
            <button 
                onClick={() => fetchVehicleTypes(pagination?.current_page || 1)} 
                className="btn btn-light border d-flex align-items-center justify-content-center"
                style={{ width: '40px', height: '40px', borderColor: 'var(--border-color)' }}
                title="Refresh List"
            >
                <RefreshCw size={18} className={loading ? "animate-spin" : ""} />
            </button>
          </div>
        </div>

        {/* Table */}
        <div className="table-responsive">
          <table className="table mb-0 align-middle">
            <thead style={{ backgroundColor: 'var(--hover-bg)' }}>
              <tr>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Vehicle Type</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Type</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Seats</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Base Fare</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Dist. Charge</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Time Charge</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom text-end" style={{ borderColor: 'var(--border-color)' }}>Actions</th>
              </tr>
            </thead>
            <tbody>
              {loading ? (
                 <tr>
                    <td colSpan={7} className="text-center py-5">
                       <div className="spinner-border text-primary" role="status">
                           <span className="visually-hidden">Loading...</span>
                       </div>
                    </td>
                 </tr>
              ) : vehicleTypes.length === 0 ? (
                <tr>
                   <td colSpan={7} className="text-center py-5 text-muted">
                      No vehicle types found.
                   </td>
                </tr>
              ) : (
                vehicleTypes.map((item) => (
                  <tr key={item.id} style={{ borderBottom: '1px solid var(--border-color)' }}>
                    <td className="px-4 py-3">
                      <div className="d-flex align-items-center gap-3">
                        <div className="rounded-3 d-flex align-items-center justify-content-center overflow-hidden position-relative" style={{ 
                            width: '50px', 
                            height: '50px', 
                            backgroundColor: 'var(--hover-bg)',
                            color: 'var(--muted-text)',
                            border: '1px solid var(--border-color)'
                        }}>
                          {item.mediaurls?.images?.[0]?.default ? (
                            <img src={item.mediaurls.images[0].default} alt={item.title} className="w-100 h-100 object-fit-cover position-absolute top-0 start-0" />
                          ) : (
                            <Car size={24} />
                          )}
                        </div>
                        <div>
                          <p className="fw-bold mb-0" style={{ color: 'var(--text-color)' }}>{item.title}</p>
                          <p className="small mb-0 text-muted-custom">ID: #{item.id}</p>
                        </div>
                      </div>
                    </td>
                    <td className="px-4 py-3">
                       <span className="badge rounded-pill border fw-normal text-capitalize" style={{ 
                           backgroundColor: 'var(--bg-color)', 
                           color: 'var(--text-color)',
                           borderColor: 'var(--border-color)'
                       }}>
                         {item.type || "—"}
                       </span>
                    </td>
                    <td className="px-4 py-3">
                       <span className="small text-muted-custom">{item.seats ?? "—"}</span>
                    </td>
                     <td className="px-4 py-3">
                       <span className="fw-bold text-success">${Number(item.base_fare || 0).toFixed(2)}</span>
                    </td>
                    <td className="px-4 py-3 text-muted-custom small">
                       ${Number(item.distance_charges_per_unit || 0).toFixed(2)} / km
                    </td>
                    <td className="px-4 py-3 text-muted-custom small">
                       ${Number(item.time_charges_per_minute || 0).toFixed(2)} / min
                    </td>
                    <td className="px-4 py-3 text-end">
                      <ActionMenu 
                        onView={() => router.push(`/admin/vehicle-types/${item.id}`)}
                        onEdit={() => router.push(`/admin/vehicle-types/${item.id}/edit`)}
                        onDelete={() => handleDeleteClick(item)}
                      />
                    </td>
                  </tr>
                ))
              )}
            </tbody>
          </table>
        </div>
      </div>

      {/* Pagination */}
      {pagination && pagination.total > 0 && (
         <div className="d-flex justify-content-between align-items-center mt-4">
           <div className="small text-muted-custom">
             Showing <span className="fw-bold text-dark">{pagination.from || 0}</span> to <span className="fw-bold text-dark">{pagination.to || 0}</span> of <span className="fw-bold text-dark">{pagination.total || 0}</span> results
           </div>
           <div className="d-flex gap-2 flex-wrap">
             <button
               onClick={() => fetchVehicleTypes(1)}
               disabled={pagination.current_page === 1}
               className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
               style={{ padding: '0.5rem 1rem' }}
             >
               <ChevronsLeft size={16} /> First
             </button>
             <button
               onClick={() => fetchVehicleTypes(pagination.current_page - 1)}
               disabled={pagination.current_page === 1}
               className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
               style={{ padding: '0.5rem 1rem' }}
             >
               <ChevronLeft size={16} /> Previous
             </button>
             
             <div className="d-flex align-items-center gap-1 flex-wrap">
               {pagination.links
                 ?.filter((link: any) => {
                   const pageNum = parseInt(link.label);
                   return !isNaN(pageNum) && link.label.trim() === pageNum.toString();
                 })
                 .map((link: any, index: number) => {
                   const isCurrent = link.active;
                   const pageNumber = parseInt(link.label);
                   
                   return (
                     <button
                       key={index}
                       onClick={() => fetchVehicleTypes(pageNumber)}
                       disabled={isCurrent}
                       className={`btn btn-sm ${isCurrent ? 'btn-primary' : 'btn-light border'}`}
                       style={{ minWidth: '32px', height: '32px', padding: '0 0.5rem' }}
                     >
                       {link.label}
                     </button>
                   );
                 })}
             </div>

             <button
               onClick={() => fetchVehicleTypes(pagination.current_page + 1)}
               disabled={pagination.current_page === pagination.last_page}
               className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
               style={{ padding: '0.5rem 1rem' }}
             >
               Next <ChevronRight size={16} />
             </button>
             <button
               onClick={() => fetchVehicleTypes(pagination.last_page)}
               disabled={pagination.current_page === pagination.last_page}
               className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
               style={{ padding: '0.5rem 1rem' }}
             >
               Last <ChevronsRight size={16} />
             </button>
           </div>
         </div>
      )}

      <DeleteModal 
        isOpen={deleteModalOpen}
        onClose={() => setDeleteModalOpen(false)}
        onConfirm={confirmDelete}
        itemName={getLocalizedValue(selectedItem?.name) || undefined}
        title="Delete Vehicle Type"
        description="Are you sure you want to delete this vehicle type? This action cannot be undone."
      />
    </div>
  );
}
