"use client";

import PageHeader from "@/components/admin/PageHeader";
import { 
  FileText, 
  DollarSign,
  Users,
  Loader2,
  Upload,
  Clock,
  MapPin,
  Car,
  Plus,
  X,
  CreditCard
} from "lucide-react";
import { useState } from "react";
import { VehicleTypeService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";
import Link from "next/link";
import axios from "axios";

interface MetaField {
    key: string;
    value: string;
}

export default function AdminCreateVehicleTypePage() {
  const router = useRouter();
  const [loading, setLoading] = useState(false);
  
  const [formData, setFormData] = useState({
    title: "",
    seats: 4,
    base_fare: "",
    distance_charges_per_unit: "",
    time_charges_per_minute: "",
    other_charges: "",
    type: "ride",
    image: null as File | null,
  });

  const [metaFields, setMetaFields] = useState<MetaField[]>([
      { key: "weight_capacity", value: "0" }
  ]);

  const [imagePreview, setImagePreview] = useState<string | null>(null);

  const handleImageChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      setFormData(prev => ({ ...prev, image: file }));
      setImagePreview(URL.createObjectURL(file));
    }
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
      const { name, value } = e.target;
      setFormData(prev => ({ 
          ...prev, 
          [name]: value 
      }));
  };

  // Meta Fields Handlers
  const addMetaField = () => {
      setMetaFields([...metaFields, { key: "", value: "" }]);
  };

  const removeMetaField = (index: number) => {
      const newFields = [...metaFields];
      newFields.splice(index, 1);
      setMetaFields(newFields);
  };

  const handleMetaChange = (index: number, field: 'key' | 'value', value: string) => {
      const newFields = [...metaFields];
      newFields[index][field] = value;
      setMetaFields(newFields);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      // Prepare Meta Data
      const metaObject: Record<string, any> = {};
      metaFields.forEach(field => {
          if (field.key) metaObject[field.key] = field.value;
      });

      const formDataPayload = new FormData();
      formDataPayload.append('title', formData.title);
      formDataPayload.append('seats', String(formData.seats));
      formDataPayload.append('base_fare', String(formData.base_fare));
      formDataPayload.append('distance_charges_per_unit', String(formData.distance_charges_per_unit));
      formDataPayload.append('time_charges_per_minute', String(formData.time_charges_per_minute));
      formDataPayload.append('other_charges', String(formData.other_charges));
      formDataPayload.append('type', formData.type);
      formDataPayload.append('meta', JSON.stringify(metaObject));

      if (formData.image) {
        formDataPayload.append('image', formData.image);
      }

      await VehicleTypeService.create(formDataPayload);
      
      toast.success("Vehicle type created successfully");
      router.push("/admin/vehicle-types");
    } catch (error: any) {
      console.error("Failed to create vehicle type:", error);
      toast.error(error.response?.data?.message || "Failed to create vehicle type");
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="py-2">
      <PageHeader 
        title="Create Vehicle Type" 
        description="Add a new vehicle category for ride hailing."
      />

      <div className="row justify-content-center">
        <div className="col-12">
           <form onSubmit={handleSubmit}>
            <div className="row g-4">
              
               {/* LEFT COLUMN: Main Content */}
               <div className="col-12 col-lg-8">
                  
                  {/* General Info */}
                  <div className="card-custom p-3 mb-3">
                    <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                      <FileText size={20} className="text-primary-custom" />
                      General Information
                    </h5>

                    <div className="mb-4">
                        <div className="row g-3">
                           <div className="col-12">
                              <label className="form-label small fw-bold text-muted-custom">Title</label>
                              <input 
                                type="text" 
                                name="title"
                                value={formData.title} 
                                onChange={handleChange} 
                                className="form-control" 
                                placeholder="e.g. Sedan" 
                                required 
                                style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                              />
                           </div>
                        </div>
                    </div>
                  </div>

                  {/* Pricing Configuration */}
                  <div className="card-custom p-3 mb-3">
                    <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                      <DollarSign size={20} className="text-primary-custom" />
                      Pricing Configuration
                    </h5>
                    
                    <div className="row g-3">
                        <div className="col-md-6">
                           <label className="form-label small fw-bold text-muted-custom">Base Fare</label>
                           <div className="input-group">
                              <span className="input-group-text" style={{ backgroundColor: 'var(--bg-color)', borderColor: 'var(--border-color)', color: 'var(--text-color)' }}><DollarSign size={16} /></span>
                              <input 
                                type="number" 
                                name="base_fare"
                                value={formData.base_fare}
                                onChange={handleChange}
                                className="form-control"
                                placeholder="0.00"
                                step="0.01"
                                required
                                style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                              />
                           </div>
                        </div>
                        <div className="col-md-6">
                           <label className="form-label small fw-bold text-muted-custom">Other Charges</label>
                           <div className="input-group">
                              <span className="input-group-text" style={{ backgroundColor: 'var(--bg-color)', borderColor: 'var(--border-color)', color: 'var(--text-color)' }}><CreditCard size={16} /></span>
                              <input 
                                type="number" 
                                name="other_charges"
                                value={formData.other_charges}
                                onChange={handleChange}
                                className="form-control"
                                placeholder="0.00"
                                step="0.01"
                                required
                                style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                              />
                           </div>
                        </div>
                        <div className="col-md-6">
                           <label className="form-label small fw-bold text-muted-custom">Distance Charges Per Unit</label>
                           <div className="input-group">
                              <span className="input-group-text" style={{ backgroundColor: 'var(--bg-color)', borderColor: 'var(--border-color)', color: 'var(--text-color)' }}><MapPin size={16} /></span>
                              <input 
                                type="number" 
                                name="distance_charges_per_unit"
                                value={formData.distance_charges_per_unit}
                                onChange={handleChange}
                                className="form-control"
                                placeholder="0.00"
                                step="0.01"
                                required
                                style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                              />
                           </div>
                        </div>
                        <div className="col-md-6">
                           <label className="form-label small fw-bold text-muted-custom">Time Charges Per Minute</label>
                           <div className="input-group">
                              <span className="input-group-text" style={{ backgroundColor: 'var(--bg-color)', borderColor: 'var(--border-color)', color: 'var(--text-color)' }}><Clock size={16} /></span>
                              <input 
                                type="number" 
                                name="time_charges_per_minute"
                                value={formData.time_charges_per_minute}
                                onChange={handleChange}
                                className="form-control"
                                placeholder="0.00"
                                step="0.01"
                                required
                                style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                              />
                           </div>
                        </div>
                    </div>
                  </div>

                  {/* Meta Fields */}
                  <div className="card-custom p-3 mb-3">
                     <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                        <Car size={20} className="text-primary-custom" />
                        Meta Fields
                     </h5>
                     
                     {metaFields.map((field, index) => (
                         <div key={index} className="row g-2 align-items-end mb-2">
                             <div className="col-5">
                                 <label className="form-label small fw-bold text-muted-custom">Key</label>
                                 <input 
                                    type="text" 
                                    value={field.key}
                                    onChange={(e) => handleMetaChange(index, 'key', e.target.value)}
                                    className="form-control"
                                    placeholder="e.g. weight_capacity"
                                    style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                                 />
                             </div>
                             <div className="col-6">
                                 <label className="form-label small fw-bold text-muted-custom">Value</label>
                                 <input 
                                    type="text" 
                                    value={field.value}
                                    onChange={(e) => handleMetaChange(index, 'value', e.target.value)}
                                    className="form-control"
                                    placeholder="Value"
                                    style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                                 />
                             </div>
                             <div className="col-1">
                                 <button 
                                    type="button" 
                                    onClick={() => removeMetaField(index)}
                                    className="btn btn-outline-danger btn-sm w-100 d-flex align-items-center justify-content-center"
                                    style={{ height: '38px' }}
                                 >
                                     <X size={16} />
                                 </button>
                             </div>
                         </div>
                     ))}
                     
                     <button 
                        type="button" 
                        onClick={addMetaField}
                        className="btn btn-primary btn-sm mt-3"
                     >
                         <Plus size={16} className="me-1" /> Add Meta Field
                     </button>
                  </div>
               </div>

               {/* RIGHT COLUMN: Sidebar */}
               <div className="col-12 col-lg-4">
                  {/* Image Upload */}
                  <div className="card-custom p-3 mb-3">
                    <h5 className="fw-bold mb-3" style={{ color: 'var(--text-color)' }}>Vehicle Image</h5>
                    <div className="d-flex align-items-center justify-content-center p-3 rounded-3 border-2 border-dashed transition-all hover-bg-custom position-relative overflow-hidden" 
                        style={{ borderColor: 'var(--border-color)', backgroundColor: 'var(--bg-color)', cursor: 'pointer', minHeight: '220px' }}>
                        <input type="file" accept="image/*" onChange={handleImageChange} className="position-absolute w-100 h-100 opacity-0 cursor-pointer" style={{ zIndex: 10 }} />
                        {imagePreview ? (
                            <div className="w-100 h-100 position-absolute top-0 start-0">
                                <img src={imagePreview} alt="Preview" className="w-100 h-100 object-fit-contain" />
                                <div className="position-absolute bottom-0 start-0 w-100 p-2 bg-dark bg-opacity-50 text-white text-center small">
                                    Click to change
                                </div>
                            </div>
                        ) : (
                          <div className="text-center">
                              <div className="mb-3 d-inline-flex p-3 rounded-circle shadow-sm" style={{ backgroundColor: 'var(--hover-bg)' }}>
                                  <Upload size={24} className="text-primary-custom" />
                              </div>
                              <p className="small fw-bold mb-1" style={{ color: 'var(--text-color)' }}>Drop image here</p>
                              <p className="small text-muted-custom mb-0">or click to upload</p>
                          </div>
                        )}
                    </div>
                  </div>

                  {/* Settings */}
                  <div className="card-custom p-3 mb-3">
                     <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                          <Car size={20} className="text-primary-custom" />
                          Configuration
                     </h5>

                     <div className="mb-3">
                         <label className="form-label small fw-bold text-muted-custom">Seats</label>
                         <div className="input-group">
                             <span className="input-group-text" style={{ backgroundColor: 'var(--bg-color)', borderColor: 'var(--border-color)', color: 'var(--text-color)' }}><Users size={16} /></span>
                            <input 
                              type="number" 
                              name="seats"
                              value={formData.seats}
                              onChange={handleChange}
                              className="form-control"
                              placeholder="4"
                              required
                              style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                            />
                         </div>
                     </div>

                     <div className="mb-3">
                         <label className="form-label small fw-bold text-muted-custom">Type</label>
                         <select 
                            name="type"
                            value={formData.type}
                            onChange={handleChange}
                            className="form-select"
                            style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                         >
                             <option value="ride">Ride</option>
                             <option value="intercity">Intercity</option>
                             <option value="courier">Courier</option>
                         </select>
                     </div>
                  </div>

                  {/* Actions */}
                  <div className="col-12">
                     <div className="d-flex gap-2">
                        <button type="submit" disabled={loading} className="btn btn-success fw-bold shadow-sm py-2 w-100" style={{ background: 'linear-gradient(135deg, rgb(249, 115, 22) 0%, rgb(234, 88, 12) 100%)', border: 'none' }}>
                          {loading ? <Loader2 size={18} className="spin-animation mx-auto" /> : "CREATE VEHICLE TYPE"}
                        </button>
                     </div>
                     <div className="mt-2 text-center">
                        <Link href="/admin/vehicle-types" className="btn btn-link btn-sm text-muted-custom text-decoration-none">
                            Cancel
                        </Link>
                     </div>
                  </div>

               </div>
            </div>
           </form>
        </div>
      </div>
    </div>
  );
}
