"use client";

import { useState, useEffect } from "react";
import { Search, MoreHorizontal, Shield, Wallet, BadgeCheck, Car, Store, User } from "lucide-react";
import PageHeader from "@/components/admin/PageHeader";
import ActionMenu from "@/components/admin/ActionMenu";
import DeleteModal from "@/components/admin/DeleteModal";
import { UserService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";

export default function AdminUsersPage() {
  const router = useRouter();
  const [users, setUsers] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [filter, setFilter] = useState("all");
  const [deleteModalOpen, setDeleteModalOpen] = useState(false);
  const [selectedUser, setSelectedUser] = useState<any>(null);

  const [pagination, setPagination] = useState({
    current_page: 1,
    last_page: 1,
    total: 0
  });

  useEffect(() => {
    fetchUsers(1);
  }, []);

  const fetchUsers = async (page = 1) => {
    setLoading(true);
    try {
      const response = await UserService.getAll({ page });
      // response.data SHOULD be the object containing { data: [...], meta: {...} } based on user input
      // BUT typical Laravel resources might return data inside data, so checking structure carefully.
      
      const responseData = response.data; 
      
      // Based on USER_REQUEST: top level has `data` array and `meta` object.
      // e.g. response.data = { data: [...], meta: {...}, links: {...} }
      
      setUsers(responseData.data || []);
      
      if (responseData.meta) {
          setPagination({
              current_page: responseData.meta.current_page,
              last_page: responseData.meta.last_page,
              total: responseData.meta.total
          });
      }
    } catch (error) {
      console.error("Failed to fetch users:", error);
      toast.error("Failed to fetch users");
    } finally {
      setLoading(false);
    }
  };

  const handlePageChange = (newPage: number) => {
      if (newPage > 0 && newPage <= pagination.last_page) {
          fetchUsers(newPage);
      }
  };

  const handleDeleteClick = (user: any) => {
    setSelectedUser(user);
    setDeleteModalOpen(true);
  };

  const confirmDelete = async () => {
    if (!selectedUser) return;
    try {
      await UserService.delete(selectedUser.id);
      setUsers(users.filter(u => u.id !== selectedUser.id));
      toast.success("User deleted successfully");
      setDeleteModalOpen(false);
      setSelectedUser(null);
    } catch (error) {
      console.error("Failed to delete user:", error);
      toast.error("Failed to delete user");
    }
  };

  const filteredUsers = filter === "all" 
    ? users 
    : users.filter(u => u.roles?.[0]?.name?.toLowerCase() === filter.toLowerCase());

  return (
    <div className="py-2">
      <PageHeader 
        title="User Management" 
        description="Manage customers, drivers, and vendor accounts."
        actionLabel="Add New User"
        actionLink="/admin/users/create"
      />

      <div className="card-custom overflow-hidden">
        {/* Toolbar */}
        <div className="p-4 border-bottom d-flex flex-column flex-md-row gap-3 justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
          <div className="d-flex p-1 rounded-3" style={{ backgroundColor: 'var(--bg-color)', border: '1px solid var(--border-color)' }}>
            {[
              { id: "all", label: "All Users" },
              { id: "customer", label: "Customers" },
              { id: "driver", label: "Drivers" },
              { id: "vendor", label: "Vendors" }
            ].map((tab) => (
              <button
                key={tab.id}
                onClick={() => setFilter(tab.id)}
                className={`px-3 py-1.5 rounded-2 text-capitalize fw-bold border-0 transition-all ${filter === tab.id ? 'shadow-sm' : ''}`}
                style={{
                  fontSize: '0.85rem',
                  backgroundColor: filter === tab.id ? 'var(--text-color)' : 'transparent',
                  color: filter === tab.id ? 'var(--bg-color)' : 'var(--muted-text)',
                }}
              >
                {tab.label}
              </button>
            ))}
          </div>
          
          <div className="position-relative" style={{ width: '280px' }}>
            <Search className="position-absolute top-50 translate-middle-y ms-3" size={16} style={{ color: 'var(--muted-text)' }} />
            <input 
              type="text" 
              placeholder="Search users..." 
              className="form-control ps-5 py-2"
              style={{ 
                borderRadius: '12px',
                backgroundColor: 'var(--bg-color)',
                borderColor: 'var(--border-color)',
                color: 'var(--text-color)',
                fontSize: '0.9rem'
              }}
            />
          </div>
        </div>

        {/* Table */}
        <div className="table-responsive">
          <table className="table mb-0 align-middle">
            <thead style={{ backgroundColor: 'var(--hover-bg)' }}>
              <tr>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>User</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Role</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Status</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Joined</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom text-end" style={{ borderColor: 'var(--border-color)' }}>Actions</th>
              </tr>
            </thead>
            <tbody>
              {loading ? (
                <tr>
                    <td colSpan={5} className="text-center py-5 text-muted-custom">Loading users...</td>
                </tr>
              ) : filteredUsers.length === 0 ? (
                <tr>
                    <td colSpan={5} className="text-center py-5 text-muted-custom">No users found.</td>
                </tr>
              ) : filteredUsers.map((user) => (
                <tr key={user.id} style={{ borderBottom: '1px solid var(--border-color)' }}>
                  <td className="px-4 py-3">
                    <div className="d-flex align-items-center gap-3">
                      <div className="rounded-circle d-flex align-items-center justify-content-center fw-bold text-uppercase" style={{ 
                        width: '40px', 
                        height: '40px', 
                        background: 'linear-gradient(135deg, #3b82f6 0%, #2563eb 100%)',
                        color: 'white',
                        fontSize: '0.75rem',
                        overflow: 'hidden'
                      }}>
                        {(() => {
                           // Type-safe(ish) check for image source
                           const imageUrl = user.mediaurls?.images?.[0]?.default || user.image || user.avatar;
                           return imageUrl ? (
                              <img 
                                src={imageUrl} 
                                alt={user.name} 
                                className="w-100 h-100 object-fit-cover"
                                onError={(e) => {
                                    (e.target as HTMLImageElement).style.display = 'none';
                                    (e.target as HTMLImageElement).nextElementSibling?.classList.remove('d-none');
                                }}
                              />
                           ) : (
                               <span>{user.name?.substring(0, 2) || 'UN'}</span>
                           );
                        })()}
                        <span className="d-none">{user.name?.substring(0, 2) || 'UN'}</span>
                      </div>
                      <div>
                        <p className="fw-bold mb-0" style={{ color: 'var(--text-color)' }}>{user.name || 'Unknown'}</p>
                        <p className="small mb-0 text-muted-custom">{user.email}</p>
                      </div>
                    </div>
                  </td>
                  <td className="px-4 py-3">
                    <div className="d-flex align-items-center gap-2">
                       {/* Map roles from array */}
                       <span className="small fw-semibold" style={{ color: 'var(--text-color)' }}>{user.roles?.[0]?.name || 'User'}</span>
                    </div>
                  </td>
                  <td className="px-4 py-3">
                    <span 
                      className="badge rounded-pill fw-bold border"
                      style={{
                        backgroundColor: user.status === "Active" ? 'rgba(16, 185, 129, 0.1)' : 'rgba(107, 114, 128, 0.1)',
                        color: user.status === "Active" ? '#10b981' : '#6b7280',
                        borderColor: user.status === "Active" ? 'rgba(16, 185, 129, 0.2)' : 'rgba(107, 114, 128, 0.2)',
                        padding: '0.5em 0.8em'
                      }}
                    >
                      {user.status || 'Active'}
                    </span>
                  </td>
                  <td className="px-4 py-3 small text-muted-custom font-medium">
                    {new Date(user.created_at || Date.now()).toLocaleDateString()}
                  </td>
                  <td className="px-4 py-3 text-end">
                    <ActionMenu 
                        onView={() => router.push(`/admin/users/${user.id}`)}
                        onEdit={() => router.push(`/admin/users/${user.id}/edit`)}
                        onDelete={() => handleDeleteClick(user)}
                    />
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
        
        {/* Pagination Controls */}
        <div className="p-4 border-top d-flex justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
            <span className="small text-muted-custom">
                Showing page <span className="fw-bold text-dark">{pagination.current_page}</span> of <span className="fw-bold text-dark">{pagination.last_page}</span>
            </span>
            <div className="d-flex gap-2">
                <button 
                    className="btn btn-sm btn-outline-secondary d-flex align-items-center gap-1"
                    onClick={() => handlePageChange(pagination.current_page - 1)}
                    disabled={pagination.current_page <= 1}
                >
                    Previous
                </button>
                <div className="d-flex gap-1 d-none d-md-flex">
                    {Array.from({ length: Math.min(pagination.last_page, 5) }, (_, i) => {
                         let p = i + 1;
                         if (pagination.last_page > 5 && pagination.current_page > 3) {
                             p = pagination.current_page - 2 + i;
                             if (p > pagination.last_page) p = p - pagination.last_page + (pagination.last_page - 5);
                         }
                         return p <= pagination.last_page ? p : null;
                    }).filter(Boolean).map(page => (
                        <button
                            key={page as number}
                            className={`btn btn-sm ${pagination.current_page === page ? 'btn-primary' : 'btn-light text-muted'}`}
                            onClick={() => handlePageChange(page as number)}
                        >
                            {page}
                        </button>
                    ))}
                </div>
                <button 
                    className="btn btn-sm btn-outline-secondary d-flex align-items-center gap-1"
                    onClick={() => handlePageChange(pagination.current_page + 1)}
                    disabled={pagination.current_page >= pagination.last_page}
                >
                    Next
                </button>
            </div>
        </div>
      </div>
      
      <DeleteModal 
        isOpen={deleteModalOpen}
        onClose={() => setDeleteModalOpen(false)}
        onConfirm={confirmDelete}
        itemName={selectedUser?.name || undefined}
        title="Delete User"
        description="Are you sure you want to delete this user? This will remove all their data."
      />
    </div>
  );
}

