"use client";

import PageHeader from "@/components/admin/PageHeader";
import { Save, X, User, Mail, Phone, Lock, Shield, CheckCircle, Globe, Wallet, Image as ImageIcon } from "lucide-react";
import Link from "next/link";
import { useState } from "react";
import { UserService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";

export default function CreateUserPage() {
  const router = useRouter();
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState("");
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    phone: "",
    mobile_verified: false,
    is_verified: false,
    role: "customer",
    password: "",
    confirmPassword: "",
    language: "en",
    wallet_balance: 0,
    image: null as File | null
  });

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value, type } = e.target;
    
    if (type === 'checkbox') {
      const checked = (e.target as HTMLInputElement).checked;
      setFormData(prev => ({ ...prev, [name]: checked }));
    } else if (type === 'file') {
      const file = (e.target as HTMLInputElement).files?.[0] || null;
      setFormData(prev => ({ ...prev, image: file }));
    } else {
      setFormData(prev => ({ ...prev, [name]: value }));
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (formData.password !== formData.confirmPassword) {
      setError("Passwords do not match");
      return;
    }

    setLoading(true);
    setError("");

    try {
      // Create FormData object to handle file upload
      const data = new FormData();
      data.append('name', formData.name);
      data.append('email', formData.email);
      data.append('mobile_number', formData.phone);
      data.append('password', formData.password);
      data.append('password_confirmation', formData.confirmPassword);
      data.append('roles[]', formData.role);
      data.append('active', '1');
      data.append('mobile_verified', formData.mobile_verified ? '1' : '0');
      if (formData.mobile_verified) {
        data.append('mobile_verified_at', new Date().toISOString());
      } else {
        data.append('mobile_verified_at', ''); // Send empty to clear on backend
      }
      data.append('is_verified', formData.is_verified ? '1' : '0');
      data.append('locale', formData.language);
      data.append('wallet_balance', formData.wallet_balance.toString());
      
      if (formData.image) {
        data.append('image', formData.image);
      }

      await UserService.create(data);
      toast.success("User created successfully");
      router.push("/admin/users");
    } catch (err: any) {
      console.error(err);
      toast.error(err.response?.data?.message || "Failed to create user");
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="py-2">
      <PageHeader 
        title="Create New User" 
        description="Add a new customer, driver, or vendor to the platform."
      />

      <div className="row justify-content-center">
        <div className="col-12">
          <form onSubmit={handleSubmit}>
            <div className="d-flex flex-column gap-4">
              
              {error && (
                <div className="alert alert-danger" role="alert">
                  {error}
                </div>
              )}

              {/* Main Info Section */}
              <div className="card-custom p-4">
                <h5 className="fw-bold mb-4 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                  <User size={20} className="text-primary-custom" />
                  User Details
                </h5>
                
                <div className="row g-4">
                  {/* Name */}
                  <div className="col-12 col-md-6">
                    <label className="form-label small fw-bold text-muted-custom">Name</label>
                    <div className="position-relative">
                      <User size={16} className="position-absolute top-50 start-0 translate-middle-y ms-3 text-muted-custom" />
                      <input 
                        type="text" 
                        name="name"
                        value={formData.name}
                        onChange={handleChange}
                        className="form-control ps-5" 
                        placeholder="Full Name"
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                        required
                      />
                    </div>
                  </div>

                  {/* Email */}
                  <div className="col-12 col-md-6">
                    <label className="form-label small fw-bold text-muted-custom">Email</label>
                    <div className="position-relative">
                      <Mail size={16} className="position-absolute top-50 start-0 translate-middle-y ms-3 text-muted-custom" />
                      <input 
                        type="email" 
                        name="email"
                        value={formData.email}
                        onChange={handleChange}
                        className="form-control ps-5" 
                        placeholder="email@example.com"
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                        required
                      />
                    </div>
                  </div>

                  {/* Mobile Number */}
                  <div className="col-12 col-md-6">
                    <label className="form-label small fw-bold text-muted-custom">Mobile Number</label>
                    <div className="position-relative">
                      <Phone size={16} className="position-absolute top-50 start-0 translate-middle-y ms-3 text-muted-custom" />
                      <input 
                        type="tel" 
                        name="phone"
                        value={formData.phone}
                        onChange={handleChange}
                        className="form-control ps-5" 
                        placeholder="+20..."
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                        required
                      />
                    </div>
                  </div>

                  {/* Mobile Verified */}
                  <div className="col-12 col-md-3 d-flex align-items-end mb-2">
                    <div className="form-check form-switch">
                      <input 
                        className="form-check-input" 
                        type="checkbox" 
                        name="mobile_verified"
                        id="mobile_verified"
                        checked={formData.mobile_verified}
                        onChange={handleChange}
                        style={{ cursor: 'pointer' }}
                      />
                      <label className="form-check-label small fw-bold text-muted-custom ms-2" htmlFor="mobile_verified">
                        Mobile Verified?
                      </label>
                    </div>
                  </div>

                  {/* Is Verified */}
                  <div className="col-12 col-md-3 d-flex align-items-end mb-2">
                    <div className="form-check form-switch">
                      <input 
                        className="form-check-input" 
                        type="checkbox" 
                        name="is_verified"
                        id="is_verified"
                        checked={formData.is_verified}
                        onChange={handleChange}
                        style={{ cursor: 'pointer' }}
                      />
                      <label className="form-check-label small fw-bold text-muted-custom ms-2" htmlFor="is_verified">
                        Is Verified?
                      </label>
                    </div>
                  </div>

                  {/* Image */}
                  <div className="col-12 col-md-6">
                    <label className="form-label small fw-bold text-muted-custom">Roles</label>
                    <div className="position-relative">
                       <Shield size={16} className="position-absolute top-50 start-0 translate-middle-y ms-3 text-muted-custom" />
                       <select 
                        name="role"
                        value={formData.role}
                        onChange={handleChange}
                        className="form-select ps-5"
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      >
                        <option value="customer">Customer</option>
                        <option value="driver">Driver</option>
                        <option value="vendor">Vendor</option>
                        <option value="admin">Admin</option>
                      </select>
                    </div>
                  </div>

                  {/* Language */}
                  <div className="col-12 col-md-6">
                    <label className="form-label small fw-bold text-muted-custom">Language</label>
                    <div className="position-relative">
                       <Globe size={16} className="position-absolute top-50 start-0 translate-middle-y ms-3 text-muted-custom" />
                       <select 
                        name="language"
                        value={formData.language}
                        onChange={handleChange}
                        className="form-select ps-5"
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      >
                        <option value="en">English</option>
                        <option value="ar">Arabic</option>
                      </select>
                    </div>
                  </div>

                   {/* Wallet Balance */}
                   <div className="col-12 col-md-6">
                    <label className="form-label small fw-bold text-muted-custom">Wallet Balance</label>
                    <div className="position-relative">
                      <Wallet size={16} className="position-absolute top-50 start-0 translate-middle-y ms-3 text-muted-custom" />
                      <input 
                        type="number" 
                        name="wallet_balance"
                        value={formData.wallet_balance}
                        onChange={handleChange}
                        className="form-control ps-5" 
                        placeholder="0.00"
                        min="0"
                        step="0.01"
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                  </div>

                  {/* Image */}
                  <div className="col-12 col-md-6">
                    <label className="form-label small fw-bold text-muted-custom">Image</label>
                    <div className="position-relative">
                      <ImageIcon size={16} className="position-absolute top-50 start-0 translate-middle-y ms-3 text-muted-custom" />
                      <input 
                        type="file" 
                        name="image"
                        onChange={handleChange}
                        className="form-control ps-5" 
                        accept="image/*"
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                  </div>

                </div>
              </div>

              {/* Security Section */}
              <div className="card-custom p-4">
                <h5 className="fw-bold mb-4 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                  <Lock size={20} className="text-primary-custom" />
                  Security
                </h5>
                <div className="row g-4">
                  <div className="col-12 col-md-6">
                    <label className="form-label small fw-bold text-muted-custom">Password</label>
                    <div className="position-relative">
                      <Lock size={16} className="position-absolute top-50 start-0 translate-middle-y ms-3 text-muted-custom" />
                       <input 
                        type="password" 
                        name="password"
                        value={formData.password}
                        onChange={handleChange}
                        className="form-control ps-5" 
                        placeholder="••••••••"
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                        required
                      />
                    </div>
                  </div>
                  <div className="col-12 col-md-6">
                    <label className="form-label small fw-bold text-muted-custom">Confirm Password</label>
                     <div className="position-relative">
                      <Lock size={16} className="position-absolute top-50 start-0 translate-middle-y ms-3 text-muted-custom" />
                      <input 
                        type="password" 
                        name="confirmPassword"
                        value={formData.confirmPassword}
                        onChange={handleChange}
                        className="form-control ps-5" 
                        placeholder="••••••••"
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                        required
                      />
                    </div>
                  </div>
                </div>
              </div>

              {/* Actions */}
              <div className="d-flex justify-content-end gap-3">
                <Link 
                  href="/admin/users"
                  className="btn d-flex align-items-center gap-2 fw-semibold px-4"
                  style={{ color: 'var(--muted-text)', borderColor: 'var(--border-color)' }}
                >
                  <X size={18} />
                  Cancel
                </Link>
                <button 
                  type="submit"
                  disabled={loading}
                  className="btn btn-primary d-flex align-items-center gap-2 px-4 fw-bold shadow-sm"
                  style={{ background: 'linear-gradient(135deg, #f97316 0%, #ea580c 100%)', border: 'none' }}
                >
                  <Save size={18} />
                  {loading ? "Creating..." : "Create User"}
                </button>
              </div>

            </div>
          </form>
        </div>
      </div>
    </div>
  );
}
