"use client";

import { useState, useEffect } from "react";
import PageHeader from "@/components/admin/PageHeader";
import ActionMenu from "@/components/admin/ActionMenu";
import DeleteModal from "@/components/admin/DeleteModal";
import { Search, MoreHorizontal, Star, MapPin, Store } from "lucide-react";
import { StoreService } from "@/services/api";
import { useRouter, useSearchParams } from "next/navigation";
import toast from "react-hot-toast";

export default function AdminStoresPage() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const initialFilter = searchParams.get('filter') || 'all';
  
  const [stores, setStores] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [filter, setFilter] = useState(initialFilter);
  const [deleteModalOpen, setDeleteModalOpen] = useState(false);
  const [selectedStore, setSelectedStore] = useState<any>(null);
  
  const [pagination, setPagination] = useState({
    current_page: 1,
    last_page: 1,
    total: 0
  });

  useEffect(() => {
    fetchStores(1);
  }, []);

  const fetchStores = async (page = 1) => {
    setLoading(true);
    try {
      const response = await StoreService.getAll({ page });
      // Identify response structure (assuming similar to users: data.data or data directly)
      const responseData = response.data;
      setStores(responseData.data || responseData || []);
      
      if (responseData.meta) {
          setPagination({
              current_page: responseData.meta.current_page,
              last_page: responseData.meta.last_page,
              total: responseData.meta.total
          });
      }
    } catch (error) {
      console.error("Failed to fetch stores:", error);
      toast.error("Failed to load stores");
    } finally {
      setLoading(false);
    }
  };

  const handlePageChange = (newPage: number) => {
      if (newPage > 0 && newPage <= pagination.last_page) {
          fetchStores(newPage);
      }
  };

  const handleDeleteClick = (store: any) => {
    setSelectedStore(store);
    setDeleteModalOpen(true);
  };

  const confirmDelete = async () => {
    if (!selectedStore) return;
    try {
        await StoreService.delete(selectedStore.id);
        setStores(stores.filter(s => s.id !== selectedStore.id));
        toast.success("Store deleted successfully");
        setDeleteModalOpen(false);
        setSelectedStore(null);
    } catch (error) {
        console.error("Failed to delete store:", error);
        toast.error("Failed to delete store");
    }
  };

  const filteredStores = stores.filter(store => {
    // Check active status (handle 1/0 or true/false)
    const isActive = store.active === 1 || store.active === true;
    // Check verified status (handle 1/0 or true/false)
    const isVerified = store.is_verified === 1 || store.is_verified === true;

    if (filter === "all") return true;
    if (filter === "active") return isActive && isVerified;
    if (filter === "pending") return !isVerified; // Treat unverified as pending
    if (filter === "inactive") return !isActive;
    return true;
  });

  return (
    <div className="py-2">
      <PageHeader 
        title="Store Management" 
        description="Manage vendor stores, approvals, and listings."
        actionLabel="Add New Store"
        actionLink="/admin/stores/create"
      />

      <div className="card-custom overflow-hidden">
        {/* Toolbar */}
        <div className="p-4 border-bottom d-flex flex-column flex-md-row gap-3 justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
          <div className="d-flex p-1 rounded-3" style={{ backgroundColor: 'var(--bg-color)', border: '1px solid var(--border-color)' }}>
            {["all", "active", "pending", "inactive"].map((tab) => (
              <button
                key={tab}
                onClick={() => setFilter(tab)}
                className={`px-3 py-1.5 rounded-2 text-capitalize fw-bold border-0 transition-all ${filter === tab ? 'shadow-sm' : ''}`}
                style={{
                  fontSize: '0.85rem',
                  backgroundColor: filter === tab ? 'var(--text-color)' : 'transparent',
                  color: filter === tab ? 'var(--bg-color)' : 'var(--muted-text)',
                }}
              >
                {tab}
              </button>
            ))}
          </div>
          
          <div className="position-relative" style={{ width: '280px' }}>
            <Search className="position-absolute top-50 translate-middle-y ms-3" size={16} style={{ color: 'var(--muted-text)' }} />
            <input 
              type="text" 
              placeholder="Search stores..." 
              className="form-control ps-5 py-2"
              style={{ 
                borderRadius: '12px',
                backgroundColor: 'var(--bg-color)',
                borderColor: 'var(--border-color)',
                color: 'var(--text-color)',
                fontSize: '0.9rem'
              }}
            />
          </div>
        </div>

        {/* Table */}
        <div className="table-responsive">
          <table className="table mb-0 align-middle">
            <thead style={{ backgroundColor: 'var(--hover-bg)' }}>
              <tr>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Store</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Contact Info</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Category</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Fees</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Status</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom text-end" style={{ borderColor: 'var(--border-color)' }}>Actions</th>
              </tr>
            </thead>
            <tbody>
              {loading ? (
                  <tr>
                      <td colSpan={6} className="text-center py-5 text-muted-custom">Loading stores...</td>
                  </tr>
              ) : filteredStores.length === 0 ? (
                  <tr>
                      <td colSpan={6} className="text-center py-5 text-muted-custom">No stores found.</td>
                  </tr>
              ) : filteredStores.map((store) => {
                const isActive = store.active === 1 || store.active === true;
                const isVerified = store.is_verified === 1 || store.is_verified === true;
                
                let statusLabel = 'Inactive';
                let statusColor = '#6b7280'; // gray
                let statusBg = 'rgba(107, 114, 128, 0.1)';
                let statusBorder = 'rgba(107, 114, 128, 0.2)';

                if (isVerified) {
                    statusLabel = 'Verified';
                    statusColor = '#10b981'; // green
                    statusBg = 'rgba(16, 185, 129, 0.1)';
                    statusBorder = 'rgba(16, 185, 129, 0.2)';
                } else {
                    statusLabel = 'Unverified';
                    statusColor = '#ef4444'; // red
                    statusBg = 'rgba(239, 68, 68, 0.1)';
                    statusBorder = 'rgba(239, 68, 68, 0.2)';
                }

                return (
                <tr key={store.id} style={{ borderBottom: '1px solid var(--border-color)' }}>
                  <td className="px-4 py-3">
                    <div className="d-flex align-items-center gap-3">
                      <div className="rounded-3 d-flex align-items-center justify-content-center fw-bold fs-5 shadow-sm overflow-hidden flex-shrink-0" style={{ 
                        width: '48px', 
                        height: '48px', 
                        backgroundColor: 'var(--bg-color)',
                        border: '1px solid var(--border-color)',
                        color: 'var(--text-color)'
                      }}>
                        {(() => {
                            // Enhanced Image Logic
                            const imageUrl = store.mediaurls?.images?.[0]?.default || store.image || store.logo;
                            return imageUrl ? (
                                <img 
                                    src={imageUrl} 
                                    alt={store.name} 
                                    className="w-100 h-100 object-fit-cover"
                                    onError={(e) => {
                                        (e.target as HTMLImageElement).style.display = 'none';
                                        (e.target as HTMLImageElement).nextElementSibling?.classList.remove('d-none');
                                    }}
                                />
                            ) : (
                                <span>{store.name?.substring(0, 1) || 'S'}</span>
                            );
                        })()}
                        <span className="d-none">{store.name?.substring(0, 1) || 'S'}</span>
                      </div>
                      <div>
                        <p className="fw-bold mb-0" style={{ color: 'var(--text-color)' }}>{store.name || 'Unknown Store'}</p>
                        <p className="small mb-0 text-muted-custom">{store.tagline || 'No Tagline'}</p>
                      </div>
                    </div>
                  </td>
                  <td className="px-4 py-3">
                      <div className="d-flex flex-column gap-1">
                          <span className="small fw-semibold" style={{ color: 'var(--text-color)' }}>{store.user?.email || 'N/A'}</span>
                          <span className="small text-muted-custom">{store.user?.mobile_number || 'N/A'}</span>
                      </div>
                  </td>
                  <td className="px-4 py-3">
                    <span className="badge bg-light text-dark border">
                        {store.categories?.[0]?.title || store.meta?.vendor_type || 'General'}
                    </span>
                  </td>
                  <td className="px-4 py-3">
                      <div className="d-flex flex-column small">
                          <div className="d-flex justify-content-between gap-3">
                               <span className="text-muted-custom">Delivery:</span>
                               <span className="fw-bold" style={{ color: 'var(--text-color)' }}>{store.delivery_fee ?? 0}</span>
                          </div>
                          <div className="d-flex justify-content-between gap-3">
                               <span className="text-muted-custom">Min Order:</span>
                               <span className="fw-bold" style={{ color: 'var(--text-color)' }}>{store.minimum_order ?? 0}</span>
                          </div>
                      </div>
                  </td>
                  <td className="px-4 py-3">
                    <span 
                      className="badge rounded-pill fw-bold border"
                      style={{
                        backgroundColor: statusBg,
                        color: statusColor,
                        borderColor: statusBorder,
                        padding: '0.5em 0.8em'
                      }}
                    >
                      {statusLabel}
                    </span>
                  </td>
                  <td className="px-4 py-3 text-end">
                    <ActionMenu 
                        onView={() => router.push(`/admin/stores/${store.id}`)}
                        onEdit={() => router.push(`/admin/stores/${store.id}/edit`)}
                        onDelete={() => handleDeleteClick(store)}
                    />
                  </td>
                </tr>
              )})}
            </tbody>
          </table>
        </div>

        {/* Pagination Controls */}
        <div className="p-4 border-top d-flex justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
            <span className="small text-muted-custom">
                Showing page <span className="fw-bold text-dark">{pagination.current_page}</span> of <span className="fw-bold text-dark">{pagination.last_page}</span>
            </span>
            <div className="d-flex gap-2">
                <button 
                    className="btn btn-sm btn-outline-secondary d-flex align-items-center gap-1"
                    onClick={() => handlePageChange(pagination.current_page - 1)}
                    disabled={pagination.current_page <= 1}
                >
                    Previous
                </button>
                <div className="d-flex gap-1 d-none d-md-flex">
                    {Array.from({ length: Math.min(pagination.last_page, 5) }, (_, i) => {
                         let p = i + 1;
                         if (pagination.last_page > 5 && pagination.current_page > 3) {
                             p = pagination.current_page - 2 + i;
                             if (p > pagination.last_page) p = p - pagination.last_page + (pagination.last_page - 5);
                         }
                         return p <= pagination.last_page ? p : null;
                    }).filter(Boolean).map(page => (
                        <button
                            key={page as number}
                            className={`btn btn-sm ${pagination.current_page === page ? 'btn-primary' : 'btn-light text-muted'}`}
                            onClick={() => handlePageChange(page as number)}
                        >
                            {page}
                        </button>
                    ))}
                </div>
                <button 
                    className="btn btn-sm btn-outline-secondary d-flex align-items-center gap-1"
                    onClick={() => handlePageChange(pagination.current_page + 1)}
                    disabled={pagination.current_page >= pagination.last_page}
                >
                    Next
                </button>
            </div>
        </div>
      </div>
      
      <DeleteModal 
        isOpen={deleteModalOpen}
        onClose={() => setDeleteModalOpen(false)}
        onConfirm={confirmDelete}
        itemName={selectedStore?.name || undefined}
        title="Delete Store"
        description="Are you sure you want to delete this store? This will also remove all associated products."
      />
    </div>
  );
}
