"use client";

import PageHeader from "@/components/admin/PageHeader";
import { Save, X, Upload, Store, MapPin, Tag, FileText, Globe, Plus, Trash2, CheckCircle } from "lucide-react";
import Link from "next/link";
import { useState, useEffect } from "react";
import { StoreService, CategoryService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";

interface Category {
  id: number;
  title: string;
  slug: string;
}

interface MetaField {
  key: string;
  value: string;
}

export default function CreateStorePage() {
  const router = useRouter();
  const [loading, setLoading] = useState(false);
  const [categories, setCategories] = useState<Category[]>([]);
  const [imagePreview, setImagePreview] = useState<string | null>(null);
  
  // Form State
  const [formData, setFormData] = useState({
    name_en: "",
    name_ar: "",
    tagline_en: "",
    tagline_ar: "",
    details_en: "",
    details_ar: "",
    // Owner Info
    email: "",
    mobile_number: "",
    password: "",
    
    category_id: "",
    minimum_order: "0",
    delivery_fee: "0",
    area: "",
    address: "",
    latitude: "",
    longitude: "",
    is_verified: false,
    image: null as File | null
  });

  const [metaFields, setMetaFields] = useState<MetaField[]>([
    { key: "discount", value: "0" },
    { key: "vendor_type", value: "grocery" }
  ]);

  useEffect(() => {
    fetchCategories();
  }, []);

  const fetchCategories = async () => {
    try {
      const response = await CategoryService.getAll();
      // Adjust based on actual API response structure (e.g. response.data.data or response.data)
      setCategories(response.data.data || response.data || []); 
    } catch (err) {
      console.error("Failed to fetch categories", err);
      toast.error("Failed to load categories");
    }
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const { name, value, type } = e.target;
    
    if (type === 'checkbox') {
        setFormData(prev => ({
            ...prev,
            [name]: (e.target as HTMLInputElement).checked
        }));
    } else {
        // @ts-ignore
        setFormData(prev => ({
            ...prev,
            [name]: value
        }));
    }
  };

  const handleImageChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      setFormData(prev => ({ ...prev, image: file }));
      setImagePreview(URL.createObjectURL(file));
    }
  };

  const handleMetaChange = (index: number, field: 'key' | 'value', value: string) => {
    const newMeta = [...metaFields];
    newMeta[index][field] = value;
    setMetaFields(newMeta);
  };

  const addMetaField = () => {
    setMetaFields([...metaFields, { key: "", value: "" }]);
  };

  const removeMetaField = (index: number) => {
    setMetaFields(metaFields.filter((_, i) => i !== index));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      // Build translation objects that controller expects (_name, _tagline, _details)
      const nameObj: Record<string, string> = {};
      if (formData.name_en) nameObj.en = formData.name_en;
      if (formData.name_ar) nameObj.ar = formData.name_ar;
      
      const taglineObj: Record<string, string> = {};
      if (formData.tagline_en) taglineObj.en = formData.tagline_en;
      if (formData.tagline_ar) taglineObj.ar = formData.tagline_ar;
      
      const detailsObj: Record<string, string> = {};
      if (formData.details_en) detailsObj.en = formData.details_en;
      if (formData.details_ar) detailsObj.ar = formData.details_ar;

      // Build meta object
      const metaObject = metaFields.reduce((acc, field) => {
        if (field.key) acc[field.key] = field.value;
        return acc;
      }, {} as Record<string, string>);

      // Create store with data that satisfies both validator and controller
      // Send translations in BOTH formats to satisfy validation AND controller
      const nameTranslations = [];
      if (formData.name_en) nameTranslations.push({ language: "en", text: formData.name_en });
      if (formData.name_ar) nameTranslations.push({ language: "ar", text: formData.name_ar });
      
      const taglineTranslations = [];
      if (formData.tagline_en) taglineTranslations.push({ language: "en", text: formData.tagline_en });
      if (formData.tagline_ar) taglineTranslations.push({ language: "ar", text: formData.tagline_ar });
      
      const detailsTranslations = [];
      if (formData.details_en) detailsTranslations.push({ language: "en", text: formData.details_en });
      if (formData.details_ar) detailsTranslations.push({ language: "ar", text: formData.details_ar });

      const storeData: any = {
        // For validation
        name_translations: nameTranslations,
        tagline_translations: taglineTranslations.length > 0 ? taglineTranslations : undefined,
        details_translations: detailsTranslations.length > 0 ? detailsTranslations : undefined,
        // For controller (what it actually uses)
        _name: nameObj,
        _tagline: Object.keys(taglineObj).length > 0 ? taglineObj : undefined,
        _details: Object.keys(detailsObj).length > 0 ? detailsObj : undefined,
        email: formData.email,
        mobile_number: formData.mobile_number,
        password: formData.password,
        categories: [formData.category_id],
        minimum_order: formData.minimum_order,
        delivery_fee: formData.delivery_fee,
        area: formData.area,
        address: formData.address,
        is_verified: formData.is_verified,
      };

      // Add optional lat/long
      if (formData.latitude && formData.latitude.toString().trim() !== "") {
        storeData.latitude = parseFloat(formData.latitude.toString());
      }
      if (formData.longitude && formData.longitude.toString().trim() !== "") {
        storeData.longitude = parseFloat(formData.longitude.toString());
      }

      // Add meta as JSON string
      if (Object.keys(metaObject).length > 0) {
        storeData.meta = JSON.stringify(metaObject);
      }

      console.log("=== Sending Data (Bypassing Validator) ===");
      console.log(JSON.stringify(storeData, null, 2));

      // Import axios to send with explicit JSON content type
      const axios = (await import('axios')).default;
      const API_URL = process.env.NEXT_PUBLIC_API_URL || 'https://admin.captaincity.net/api';
      const token = localStorage.getItem('token');
      
      // Send as JSON with explicit headers
      const response = await axios.post(`${API_URL}/admin/vendors`, storeData, {
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'Authorization': token ? `Bearer ${token}` : '',
        },
      });
      
      // Step 2: Upload image if exists
      if (formData.image && response.data?.data?.id) {
        const imageData = new FormData();
        imageData.append("image", formData.image);
        imageData.append("_method", "PUT");
        await axios.post(`${API_URL}/admin/vendors/${response.data.data.id}`, imageData, {
          headers: {
            'Authorization': token ? `Bearer ${token}` : '',
          },
        });
      }

      toast.success("Store created successfully!");
      router.push("/admin/stores");
    } catch (err: any) {
      console.error(err);
      toast.error(err.response?.data?.message || "Failed to create store");
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="py-2">
      <PageHeader 
        title="Create New Store" 
        description="Add a new vendor store with detailed information."
      />

      <div className="row justify-content-center">
        <div className="col-12">
          <form onSubmit={handleSubmit}>
            <div className="row g-4">
              
              {/* Main Info Column */}
              <div className="col-12 col-lg-8">
                <div className="card-custom p-3 mb-3">
                  <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                    <Globe size={20} className="text-primary-custom" />
                    Basic Information (Localized)
                  </h5>
                  
                  <div className="row g-3">
                    {/* Name */}
                    <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Name (English)</label>
                      <input 
                        type="text" 
                        name="name_en"
                        value={formData.name_en}
                        onChange={handleChange}
                        className="form-control" 
                        required
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                     <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Name (Translation/Arabic)</label>
                      <input 
                        type="text" 
                        name="name_ar"
                        value={formData.name_ar}
                        onChange={handleChange}
                        className="form-control" 
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>

                    {/* Tagline */}
                     <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Tagline (English)</label>
                      <input 
                        type="text" 
                        name="tagline_en"
                        value={formData.tagline_en}
                        onChange={handleChange}
                        className="form-control" 
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                     <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Tagline (Translation/Arabic)</label>
                      <input 
                        type="text" 
                        name="tagline_ar"
                        value={formData.tagline_ar}
                        onChange={handleChange}
                        className="form-control" 
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>

                    {/* Details */}
                    <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Details (English)</label>
                      <textarea 
                        name="details_en"
                        value={formData.details_en}
                        onChange={handleChange}
                        className="form-control" 
                        rows={2}
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                     <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Details (Translation/Arabic)</label>
                      <textarea 
                        name="details_ar"
                        value={formData.details_ar}
                        onChange={handleChange}
                        className="form-control" 
                        rows={2}
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                  </div>
                </div>

                {/* Owner Information */}
                <div className="card-custom p-3 mb-3">
                  <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                    <Globe size={20} className="text-primary-custom" />
                    Owner Information
                  </h5>
                  
                  <div className="row g-3">
                    <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Email Address</label>
                      <input 
                        type="email" 
                        name="email"
                        value={formData.email}
                        onChange={handleChange}
                        className="form-control" 
                        required
                        autoComplete="new-password"
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Mobile Number</label>
                      <input 
                        type="text" 
                        name="mobile_number"
                        value={formData.mobile_number}
                        onChange={handleChange}
                        className="form-control" 
                        required
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Password</label>
                      <input 
                        type="password" 
                        name="password"
                        value={formData.password}
                        onChange={handleChange}
                        className="form-control" 
                        required
                        autoComplete="new-password"
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                  </div>
                </div>

                {/* Location & Settings */}
                <div className="card-custom p-3 mb-3">
                   <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                    <MapPin size={20} className="text-primary-custom" />
                    Location & Settings
                  </h5>

                  <div className="row g-3">
                     {/* Category & Verified */}
                     <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Category</label>
                      <select 
                        name="category_id"
                        value={formData.category_id}
                        onChange={handleChange}
                        className="form-select"
                        required
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      >
                        <option value="">Select Category...</option>
                        {categories.map(cat => (
                            <option key={cat.id} value={cat.id}>{cat.title}</option>
                        ))}
                      </select>
                    </div>
                    <div className="col-12 col-md-6 d-flex align-items-center">
                        <div className="form-check form-switch p-0 m-0 d-flex align-items-center gap-2 mt-4">
                           <input 
                            className="form-check-input ms-0" 
                            type="checkbox" 
                            role="switch" 
                            id="is_verified"
                            name="is_verified"
                            checked={formData.is_verified}
                            onChange={handleChange}
                            style={{ 
                              width: '3em', 
                              height: '1.5em', 
                              cursor: 'pointer',
                              backgroundColor: formData.is_verified ? '#10b981' : 'var(--border-color)',
                              borderColor: formData.is_verified ? '#10b981' : 'var(--border-color)' 
                            }} 
                           />
                           <label className="form-check-label fw-bold text-muted-custom cursor-pointer" htmlFor="is_verified">
                             Is Verified?
                           </label>
                        </div>
                    </div>

                    {/* Fees */}
                    <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Minimum Order</label>
                      <input 
                        type="number" 
                        name="minimum_order"
                        value={formData.minimum_order}
                        onChange={handleChange}
                        className="form-control" 
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                     <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Delivery Fee</label>
                      <input 
                        type="number" 
                        name="delivery_fee"
                        value={formData.delivery_fee}
                        onChange={handleChange}
                        className="form-control" 
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>

                    {/* Address */}
                    <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Area</label>
                      <input 
                        type="text" 
                        name="area"
                        value={formData.area}
                        onChange={handleChange}
                        className="form-control" 
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                     <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Address</label>
                      <input 
                        type="text" 
                        name="address"
                        value={formData.address}
                        onChange={handleChange}
                        className="form-control" 
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                     <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Latitude</label>
                      <input 
                        type="number" 
                        step="any"
                        name="latitude"
                        value={formData.latitude}
                        onChange={handleChange}
                        placeholder="e.g., 30.540917"
                        className="form-control" 
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                     <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Longitude</label>
                      <input 
                        type="number" 
                        step="any"
                        name="longitude"
                        value={formData.longitude}
                        onChange={handleChange}
                        placeholder="e.g., 31.366250"
                        className="form-control" 
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                  </div>
                </div>

                 {/* Meta Fields */}
                 <div className="card-custom p-3">
                   <div className="d-flex justify-content-between align-items-center mb-3">
                     <h5 className="fw-bold mb-0 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                        <FileText size={20} className="text-primary-custom" />
                        Meta Fields
                     </h5>
                     <button type="button" onClick={addMetaField} className="btn btn-sm btn-outline-primary fw-bold" style={{ borderColor: 'var(--border-color)', color: 'var(--text-color)' }}>
                        <Plus size={16} /> Add Meta
                     </button>
                   </div>
                   
                   <div className="d-flex flex-column gap-2">
                     {metaFields.map((field, index) => (
                        <div key={index} className="row g-2 align-items-center">
                            <div className="col-5">
                                <label className="small fw-bold text-muted-custom mb-1">Key</label>
                                <input 
                                    type="text" 
                                    className="form-control form-control-sm"
                                    value={field.key}
                                    onChange={(e) => handleMetaChange(index, 'key', e.target.value)}
                                    style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                                />
                            </div>
                             <div className="col-6">
                                <label className="small fw-bold text-muted-custom mb-1">Value</label>
                                <input 
                                    type="text" 
                                    className="form-control form-control-sm"
                                    value={field.value}
                                    onChange={(e) => handleMetaChange(index, 'value', e.target.value)}
                                    style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                                />
                            </div>
                            <div className="col-1 d-flex align-items-end justify-content-center pb-1">
                                <button type="button" onClick={() => removeMetaField(index)} className="btn btn-sm text-danger p-0 border-0">
                                    <X size={18} />
                                </button>
                            </div>
                        </div>
                     ))}
                   </div>
                   <div className="mt-3">
                       <button type="button" className="btn btn-primary w-100 fw-bold shadow-sm" style={{ background: 'linear-gradient(135deg, rgb(249, 115, 22) 0%, rgb(234, 88, 12) 100%)', border: 'none' }}>
                            ADD METAFIELDS
                       </button>
                   </div>
                </div>
              </div>

              {/* Sidebar Column: Image & Actions */}
              <div className="col-12 col-lg-4">
                
                {/* Image Upload */}
                <div className="card-custom p-3 mb-3">
                  <h5 className="fw-bold mb-3" style={{ color: 'var(--text-color)' }}>Store Image/Logo</h5>
                  
                  <div className="d-flex flex-column gap-2">
                    <label className="form-label small fw-bold text-muted-custom">Image File</label>
                    <div className="d-flex align-items-center justify-content-center p-3 rounded-3 border-2 border-dashed transition-all hover-bg-custom position-relative overflow-hidden" 
                      style={{ 
                        borderColor: 'var(--border-color)', 
                        backgroundColor: 'var(--bg-color)',
                        cursor: 'pointer',
                        minHeight: '200px'
                      }}
                    >
                      <input 
                         type="file" 
                         accept="image/*"
                         onChange={handleImageChange}
                         className="position-absolute w-100 h-100 opacity-0 cursor-pointer"
                         style={{ zIndex: 10 }}
                      />
                      {imagePreview ? (
                          <div className="w-100 h-100 position-absolute top-0 start-0">
                              <img src={imagePreview} alt="Preview" className="w-100 h-100 object-fit-cover" />
                              <div className="position-absolute bottom-0 start-0 w-100 p-2 bg-dark bg-opacity-50 text-white text-center small">
                                  Click to change
                              </div>
                          </div>
                      ) : (
                        <div className="text-center">
                            <div className="mb-3 d-inline-flex p-3 rounded-circle shadow-sm" style={{ backgroundColor: 'var(--hover-bg)' }}>
                            <Upload size={24} className="text-primary-custom" />
                            </div>
                            <p className="small fw-bold mb-1" style={{ color: 'var(--text-color)' }}>Choose File</p>
                            <p className="small text-muted-custom mb-0">No file chosen</p>
                        </div>
                      )}
                    </div>
                  </div>
                </div>

                {/* More Actions Panel (Mock) */}
                 <div className="card-custom p-4">
                    <h5 className="fw-bold mb-3" style={{ color: '#3b82f6' }}>More Actions</h5>
                    <div className="d-flex gap-2">
                        <button type="button" className="btn btn-primary fw-bold text-white flex-grow-1" style={{ background: 'linear-gradient(135deg, rgb(249, 115, 22) 0%, rgb(234, 88, 12) 100%)', border: 'none', fontSize: '0.8rem' }}>
                            CREATE NEW PRODUCT
                        </button>
                         <button type="button" className="btn btn-primary fw-bold text-white flex-grow-1" style={{ background: 'linear-gradient(135deg, rgb(249, 115, 22) 0%, rgb(234, 88, 12) 100%)', border: 'none', fontSize: '0.8rem' }}>
                            VIEW ALL PRODUCTS
                        </button>
                    </div>
                 </div>

              </div>

              {/* Actions Footer */}
              <div className="col-12">
                 <div className="d-flex gap-3 mt-2">
                    <button 
                      type="submit"
                      disabled={loading}
                      className="btn btn-success d-flex align-items-center gap-2 px-4 fw-bold shadow-sm"
                      style={{ background: 'linear-gradient(135deg, rgb(249, 115, 22) 0%, rgb(234, 88, 12) 100%)', border: 'none' }}
                    >
                      <Save size={18} />
                      {loading ? "SAVING..." : "SAVE"}
                    </button>
                    <Link 
                      href="/admin/stores"
                      className="btn btn-danger d-flex align-items-center gap-2 fw-semibold px-4"
                      style={{ background: '#ef4444', border: 'none' }}
                    >
                      <X size={18} />
                      BACK
                    </Link>
                  </div>
              </div>

            </div>
          </form>
        </div>
      </div>
    </div>
  );
}
