"use client";

import PageHeader from "@/components/admin/PageHeader";
import ActionMenu from "@/components/admin/ActionMenu";
import DeleteModal from "@/components/admin/DeleteModal";
import { Search, Store, RefreshCw, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight, CheckCircle, XCircle } from "lucide-react";
import { useState, useEffect } from "react";
import { ProviderService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";

export default function AdminProvidersPage() {
  const router = useRouter();
  const [providers, setProviders] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [deleteModalOpen, setDeleteModalOpen] = useState(false);
  const [selectedItem, setSelectedItem] = useState<any>(null);
  const [page, setPage] = useState(1);
  const [pagination, setPagination] = useState<any>(null);

  useEffect(() => {
    fetchProviders(page);
  }, [page]);

  const fetchProviders = async (page = 1) => {
    setLoading(true);
    try {
      const params: any = { page };
      if (searchTerm) {
        params.search = searchTerm;
      }
      const response = await ProviderService.getAll(params);
      const resData = response.data;
      setProviders(resData.data || []);
      setPagination(resData.meta || resData);
    } catch (error) {
      console.error("Failed to fetch providers:", error);
      toast.error("Failed to load providers");
    } finally {
      setLoading(false);
    }
  };

  const handleDeleteClick = (item: any) => {
    setSelectedItem(item);
    setDeleteModalOpen(true);
  };

  const confirmDelete = async () => {
    if (!selectedItem) return;
    try {
      await ProviderService.delete(selectedItem.id);
      toast.success("Provider deleted successfully");
      setDeleteModalOpen(false);
      setSelectedItem(null);
      fetchProviders(page);
    } catch (error) {
      console.error("Failed to delete provider:", error);
      toast.error("Failed to delete provider");
    }
  };

  return (
    <div className="py-2">
      <PageHeader 
        title="Providers" 
        description="Manage service providers and vendors."
        actionLabel="Add New Provider"
        actionLink="/admin/providers/create"
      />

      <div className="card-custom overflow-hidden">
        {/* Toolbar */}
        <div className="p-4 border-bottom d-flex flex-column flex-md-row gap-3 justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
          <div className="d-flex p-1 rounded-3" style={{ backgroundColor: 'var(--bg-color)', border: '1px solid var(--border-color)' }}>
             <button
               className="px-3 py-1.5 rounded-2 text-capitalize fw-bold border-0 transition-all text-nowrap shadow-sm"
               style={{
                 fontSize: '0.85rem',
                 backgroundColor: 'var(--text-color)',
                 color: 'var(--bg-color)',
               }}
             >
               All Providers
             </button>
          </div>

          <div className="d-flex gap-2 w-100 w-md-auto">
             <div className="position-relative flex-grow-1" style={{ width: '280px' }}>
                <Search className="position-absolute top-50 translate-middle-y ms-3" size={16} style={{ color: 'var(--muted-text)' }} />
                <input 
                  type="text" 
                  placeholder="Search providers..." 
                  value={searchTerm}
                  onChange={(e) => { setSearchTerm(e.target.value); setPage(1); }}
                  onKeyDown={(e) => { if (e.key === 'Enter') fetchProviders(1); }}
                  className="form-control ps-5 py-2"
                  style={{ 
                    borderRadius: '12px',
                    backgroundColor: 'var(--bg-color)',
                    borderColor: 'var(--border-color)',
                    color: 'var(--text-color)',
                    fontSize: '0.9rem'
                  }}
                />
            </div>
            <button 
                onClick={() => fetchProviders(pagination?.current_page || 1)} 
                className="btn btn-light border d-flex align-items-center justify-content-center"
                style={{ width: '40px', height: '40px', borderColor: 'var(--border-color)' }}
                title="Refresh List"
            >
                <RefreshCw size={18} className={loading ? "animate-spin" : ""} />
            </button>
          </div>
        </div>

        {/* Table */}
        <div className="table-responsive">
          <table className="table mb-0 align-middle">
            <thead style={{ backgroundColor: 'var(--hover-bg)' }}>
              <tr>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Provider</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Contact Info</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Categories</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Address</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Status</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Joined</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom text-end" style={{ borderColor: 'var(--border-color)' }}>Actions</th>
              </tr>
            </thead>
            <tbody>
              {loading ? (
                 <tr>
                    <td colSpan={7} className="text-center py-5">
                       <div className="spinner-border text-primary" role="status">
                           <span className="visually-hidden">Loading...</span>
                       </div>
                    </td>
                 </tr>
              ) : providers.length === 0 ? (
                <tr>
                   <td colSpan={7} className="text-center py-5 text-muted">
                      No providers found.
                   </td>
                </tr>
              ) : (
                providers.map((item) => (
                  <tr key={item.id} style={{ borderBottom: '1px solid var(--border-color)' }}>
                    <td className="px-4 py-3">
                      <div className="d-flex align-items-center gap-3">
                        <div className="position-relative">
                          {item.user?.mediaurls?.images?.[0]?.default || item.mediaurls?.images?.[0]?.default ? (
                            <img 
                              src={item.user?.mediaurls?.images?.[0]?.default || item.mediaurls?.images?.[0]?.default} 
                              alt={item.name}
                              className="rounded-circle"
                              style={{ 
                                width: '48px', 
                                height: '48px', 
                                objectFit: 'cover',
                                border: '2px solid var(--border-color)'
                              }}
                            />
                          ) : (
                            <div className="rounded-circle d-flex align-items-center justify-content-center" style={{ 
                                width: '48px', 
                                height: '48px', 
                                backgroundColor: 'var(--hover-bg)',
                                color: 'var(--primary-color)',
                                border: '1px solid var(--border-color)'
                            }}>
                              <Store size={22} />
                            </div>
                          )}
                          {item.ratings > 0 && (
                            <span className="position-absolute -bottom-1 -end-1 bg-warning text-white text-xs px-1 rounded-pill d-flex align-items-center shadow-sm" style={{ fontSize: '0.65rem', border: '1px solid #fff' }}>
                              ★ {item.ratings}
                            </span>
                          )}
                        </div>
                        <div>
                          <p className="fw-bold mb-0 text-dark" style={{ fontSize: '0.95rem' }}>{item.name || 'N/A'}</p>
                          <p className="small mb-0 text-muted-custom">ID: #{item.id}</p>
                        </div>
                      </div>
                    </td>
                    <td className="px-4 py-3">
                       <div className="d-flex flex-column gap-1">
                         <div className="d-flex align-items-center gap-2 small text-muted-custom">
                            <Store size={14} />
                            <span className="text-truncate" style={{ maxWidth: '150px' }}>{item.user?.email || item.email || '—'}</span>
                         </div>
                         <div className="d-flex align-items-center gap-2 small text-muted-custom">
                            <Store size={14} className="text-muted-custom" style={{ opacity: 0.7 }} /> 
                           {/* Using Store icon as placeholder, should be Phone but imports are limited in this replacement block context if I don't add it. 
                               Wait, I can't add imports easily here. Using existing Store icon or just text. 
                               Actually I can see imports at top of file, Phone is not imported? 
                               Let's check imports. Store is imported. Phone is NOT imported in original file context I see. 
                               Wait, I see Phone in ShowProviderPage but not necessarily here. 
                               Let's stick to simple text for safety or basic icons. */}
                            <span>{item.user?.mobile_number || item.mobile || '—'}</span>
                         </div>
                       </div>
                    </td>
                    <td className="px-4 py-3">
                       <div className="d-flex flex-wrap gap-1" style={{ maxWidth: '200px' }}>
                         {item.categories && item.categories.length > 0 ? (
                           item.categories.slice(0, 3).map((cat: any) => (
                             <span key={cat.category_id} className="badge bg-light text-dark border small fw-normal">
                               {cat.category?.title || 'Category'}
                             </span>
                           ))
                         ) : (
                           <span className="small text-muted-custom">—</span>
                         )}
                         {item.categories && item.categories.length > 3 && (
                           <span className="badge bg-light text-muted border small fw-normal">
                             +{item.categories.length - 3}
                           </span>
                         )}
                       </div>
                    </td>
                    <td className="px-4 py-3">
                       <span className="small text-muted-custom d-block text-truncate" style={{ maxWidth: '150px' }} title={item.address}>
                         {item.address || '—'}
                       </span>
                    </td>
                    <td className="px-4 py-3">
                       {item.is_verified ? (
                         <span className="badge bg-success-subtle text-success border border-success-subtle rounded-pill d-inline-flex align-items-center gap-1 px-2 py-1">
                           <CheckCircle size={12} /> Verified
                         </span>
                       ) : (
                         <span className="badge bg-warning-subtle text-warning border border-warning-subtle rounded-pill d-inline-flex align-items-center gap-1 px-2 py-1">
                           <XCircle size={12} /> Pending
                         </span>
                       )}
                    </td>
                    <td className="px-4 py-3">
                       <span className="small text-muted-custom">{new Date(item.created_at).toLocaleDateString()}</span>
                    </td>
                    <td className="px-4 py-3 text-end">
                      <ActionMenu 
                        onView={() => router.push(`/admin/providers/${item.id}`)}
                        onEdit={() => router.push(`/admin/providers/${item.id}/edit`)}
                        onDelete={() => handleDeleteClick(item)}
                      />
                    </td>
                  </tr>
                ))
              )}
            </tbody>
          </table>
        </div>
      </div>

      {/* Pagination */}
      {pagination && pagination.total > 0 && (
         <div className="d-flex justify-content-between align-items-center mt-4">
           <div className="small text-muted-custom">
             Showing <span className="fw-bold text-dark">{pagination.from || 0}</span> to <span className="fw-bold text-dark">{pagination.to || 0}</span> of <span className="fw-bold text-dark">{pagination.total || 0}</span> results
           </div>
           <div className="d-flex gap-2 flex-wrap">
             <button
               onClick={() => fetchProviders(1)}
               disabled={pagination.current_page === 1}
               className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
               style={{ padding: '0.5rem 1rem' }}
             >
               <ChevronsLeft size={16} /> First
             </button>
             <button
               onClick={() => fetchProviders(pagination.current_page - 1)}
               disabled={pagination.current_page === 1}
               className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
               style={{ padding: '0.5rem 1rem' }}
             >
               <ChevronLeft size={16} /> Previous
             </button>
             
             <div className="d-flex align-items-center gap-1 flex-wrap">
               {pagination.links
                 ?.filter((link: any) => {
                   const pageNum = parseInt(link.label);
                   return !isNaN(pageNum) && link.label.trim() === pageNum.toString();
                 })
                 .map((link: any, index: number) => {
                   const isCurrent = link.active;
                   const pageNumber = parseInt(link.label);
                   
                   return (
                     <button
                       key={index}
                       onClick={() => fetchProviders(pageNumber)}
                       disabled={isCurrent}
                       className={`btn btn-sm ${isCurrent ? 'btn-primary' : 'btn-light border'}`}
                       style={{ minWidth: '32px', height: '32px', padding: '0 0.5rem' }}
                     >
                       {link.label}
                     </button>
                   );
                 })}
             </div>

             <button
               onClick={() => fetchProviders(pagination.current_page + 1)}
               disabled={pagination.current_page === pagination.last_page}
               className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
               style={{ padding: '0.5rem 1rem' }}
             >
               Next <ChevronRight size={16} />
             </button>
             <button
               onClick={() => fetchProviders(pagination.last_page)}
               disabled={pagination.current_page === pagination.last_page}
               className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
               style={{ padding: '0.5rem 1rem' }}
             >
               Last <ChevronsRight size={16} />
             </button>
           </div>
         </div>
      )}

      <DeleteModal 
        isOpen={deleteModalOpen}
        onClose={() => setDeleteModalOpen(false)}
        onConfirm={confirmDelete}
        itemName={selectedItem?.name || undefined}
        title="Delete Provider"
        description="Are you sure you want to delete this provider? This action cannot be undone."
      />
    </div>
  );
}
