"use client";

import PageHeader from "@/components/admin/PageHeader";
import { Check, X, DollarSign, Wallet } from "lucide-react";
import { useState, useEffect } from "react";
import { PayoutService } from "@/services/api";
import Pagination from "@/components/admin/Pagination";
import toast from "react-hot-toast";

export default function PayoutsPage() {
  const [payouts, setPayouts] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [pagination, setPagination] = useState<any>(null);
  const [page, setPage] = useState(1);

  useEffect(() => {
    fetchPayouts();
  }, [page]);

  const fetchPayouts = async () => {
    setLoading(true);
    try {
      const response = await PayoutService.getAll({ page });
      setPayouts(response.data.data);
      setPagination(response.data.meta);
    } catch (error) {
      console.error("Failed to fetch payouts:", error);
      toast.error("Failed to load payouts");
    } finally {
      setLoading(false);
    }
  };

  const handleStatusUpdate = async (id: number, currentStatus: boolean) => {
     try {
        const newStatus = !currentStatus;
        await PayoutService.updateStatus(id, newStatus);
        
        // Optimistic Update
        setPayouts(prev => prev.map((p: any) => 
            p.id === id ? { 
                ...p, 
                confirmed: newStatus ? 1 : 0,
                meta: { ...p.meta, is_paid: newStatus ? "1" : "0" }
            } : p
        ));
        
        toast.success(newStatus ? "Marked as Paid" : "Marked as Unpaid");
     } catch (err) {
         console.error(err);
         toast.error("Failed to update status");
     }
  };

  return (
    <div className="py-2">
      <PageHeader title="Payouts" description="Manage withdrawals and payout requests." />

      <div className="card-custom overflow-hidden">
        <div className="table-responsive">
          <table className="table table-hover align-middle mb-0">
            <thead className="bg-light">
              <tr>
                <th className="px-4 py-3 text-secondary small text-uppercase">ID</th>
                <th className="px-4 py-3 text-secondary small text-uppercase">User</th>
                <th className="px-4 py-3 text-secondary small text-uppercase">Bank Details</th>
                <th className="px-4 py-3 text-secondary small text-uppercase text-end">Amount</th>
                <th className="px-4 py-3 text-secondary small text-uppercase text-center">Is Paid</th>
                <th className="px-4 py-3 text-secondary small text-uppercase text-end">Action</th>
              </tr>
            </thead>
            <tbody>
              {loading ? (
                <tr>
                   <td colSpan={6} className="text-center py-5">
                      <div className="spinner-border text-primary" role="status">
                        <span className="visually-hidden">Loading...</span>
                      </div>
                   </td>
                </tr>
              ) : payouts.length === 0 ? (
                <tr>
                  <td colSpan={6} className="text-center py-5 text-muted">
                    No payouts found.
                  </td>
                </tr>
              ) : (
                payouts.map((item: any) => {
                  const isPaid = item.meta?.is_paid === "1" || item.confirmed === 1;
                  return (
                  <tr key={item.id}>
                    <td className="px-4 py-3 fw-medium">#{item.id}</td>
                    <td className="px-4 py-3">
                       {item.user ? (
                           <div className="d-flex align-items-center gap-3">
                               <div className="bg-light rounded-circle p-2">
                                   <UserIcon />
                               </div>
                               <div className="d-flex flex-column">
                                  <span className="fw-medium text-dark">{item.user.name || 'Unknown'}</span>
                                  <span className="small text-muted">{item.user.email}</span>
                               </div>
                           </div>
                       ) : (
                           <span className="text-muted small">Not Available</span>
                       )}
                    </td>
                    <td className="px-4 py-3">
                        <div className="d-flex flex-column small">
                            {item.meta?.bank_name && (
                                <span className="fw-bold text-dark">{item.meta.bank_name}</span>
                            )}
                            {item.meta?.bank_account_name && (
                                <span className="text-muted">{item.meta.bank_account_name}</span>
                            )}
                            {item.meta?.bank_account_number && (
                                <span className="font-monospace text-muted">{item.meta.bank_account_number}</span>
                            )}
                            {!item.meta?.bank_name && !item.meta?.bank_account_number && (
                                <span className="text-muted">-</span>
                            )}
                        </div>
                    </td>
                    <td className="px-4 py-3 text-end fw-bold fs-6 text-dark">
                        {parseFloat(item.amount).toFixed(2)}
                    </td>
                    <td className="px-4 py-3 text-center">
                        {isPaid ? (
                            <span className="badge bg-success text-white px-3 py-2 rounded-pill">YES</span>
                        ) : (
                            <span className="badge bg-danger text-white px-3 py-2 rounded-pill">NO</span>
                        )}
                    </td>
                    <td className="px-4 py-3 text-end">
                       <button 
                         onClick={() => handleStatusUpdate(item.id, isPaid)}
                         className={`btn btn-sm fw-bold px-3 ${isPaid ? 'btn-outline-danger' : 'btn-outline-success'}`}
                       >
                           {isPaid ? 'Mark Unpaid' : 'Mark Paid'}
                       </button>
                    </td>
                  </tr>
                )})
              )}
            </tbody>
          </table>
        </div>
        
        {!loading && pagination && (
            <div className="p-3 border-top">
                <Pagination 
                   currentPage={page}
                   lastPage={pagination.last_page}
                   onPageChange={setPage}
                />
            </div>
        )}
      </div>
    </div>
  );
}

function UserIcon() {
    return (
        <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="text-muted">
            <path d="M19 21v-2a4 4 0 0 0-4-4H9a4 4 0 0 0-4 4v2"></path>
            <circle cx="12" cy="7" r="4"></circle>
        </svg>
    )
}
