"use client";

import { ResponsiveContainer, AreaChart, Area, XAxis, YAxis, CartesianGrid, Tooltip, PieChart, Pie, Cell } from "recharts";
import { ArrowUpRight, ArrowDownRight, Users, DollarSign, ShoppingBag, Car, Store, Package, TrendingUp } from "lucide-react";

const REVENUE_DATA = [
  { name: "Mon", ecommerce: 4000, rides: 2400 },
  { name: "Tue", ecommerce: 3000, rides: 1398 },
  { name: "Wed", ecommerce: 5000, rides: 3800 },
  { name: "Thu", ecommerce: 2780, rides: 3908 },
  { name: "Fri", ecommerce: 6890, rides: 4800 },
  { name: "Sat", ecommerce: 8390, rides: 3800 },
  { name: "Sun", ecommerce: 7490, rides: 4300 },
];

const USER_DISTRIBUTION = [
  { name: "Customers", value: 2350, color: "#10b981" },
  { name: "Vendors", value: 234, color: "#f59e0b" },
  { name: "Drivers", value: 573, color: "#3b82f6" },
];

const STATS = [
  { label: "Total Revenue", value: "$125,231", change: "+20.1%", trend: "up", icon: DollarSign, color: "text-emerald-500", bg: "bg-emerald-500/10", border: "border-emerald-500/20" },
  { label: "Active Users", value: "3,157", change: "+12.5%", trend: "up", icon: Users, color: "text-blue-500", bg: "bg-blue-500/10", border: "border-blue-500/20" },
  { label: "Total Orders", value: "12,234", change: "+19%", trend: "up", icon: ShoppingBag, color: "text-amber-500", bg: "bg-amber-500/10", border: "border-amber-500/20" },
  { label: "Active Rides", value: "89", change: "-4%", trend: "down", icon: Car, color: "text-primary", bg: "bg-primary/10", border: "border-primary/20" },
  { label: "Active Stores", value: "234", change: "+8.2%", trend: "up", icon: Store, color: "text-purple-500", bg: "bg-purple-500/10", border: "border-purple-500/20" },
  { label: "Products Listed", value: "4,521", change: "+15%", trend: "up", icon: Package, color: "text-pink-500", bg: "bg-pink-500/10", border: "border-pink-500/20" },
];

const RECENT_ACTIVITY = [
  { type: "order", title: "New Order #12345", desc: "$45.50 • Burger Queen", time: "2m ago", icon: ShoppingBag, color: "bg-amber-500/10 text-amber-500" },
  { type: "user", title: "New User Registration", desc: "John Doe • Customer", time: "5m ago", icon: Users, color: "bg-blue-500/10 text-blue-500" },
  { type: "ride", title: "Ride Completed", desc: "$12.50 • Downtown to Airport", time: "8m ago", icon: Car, color: "bg-primary/10 text-primary" },
  { type: "store", title: "Store Approved", desc: "Fresh Mart • Grocery", time: "15m ago", icon: Store, color: "bg-emerald-500/10 text-emerald-500" },
  { type: "order", title: "Order Delivered", desc: "#12340 • Pizza Palace", time: "22m ago", icon: ShoppingBag, color: "bg-amber-500/10 text-amber-500" },
];

export default function AdminDashboard() {
  return (
    <div className="py-4">
      {/* Stats Grid */}
      <div className="row g-4 mb-5">
        {STATS.map((stat, i) => (
          <div key={i} className="col-12 col-md-6 col-lg-4">
            <div className="card-custom h-100 p-4 transition-all hover:translate-y-[-4px]" style={{ transition: 'transform 0.3s ease' }}>
              <div className="d-flex justify-content-between align-items-start mb-4">
                <div className={`rounded-3 d-flex align-items-center justify-content-center`} style={{ 
                  width: '56px', 
                  height: '56px', 
                  backgroundColor: stat.trend === 'up' ? 'rgba(16, 185, 129, 0.1)' : 'rgba(239, 68, 68, 0.1)',
                  color: stat.trend === 'up' ? '#10b981' : '#ef4444'
                }}>
                  <stat.icon size={24} />
                </div>
                <div className={`px-2 py-1 rounded-pill d-flex align-items-center gap-1 small fw-bold`} style={{
                  backgroundColor: stat.trend === "up" ? "rgba(16, 185, 129, 0.1)" : "rgba(239, 68, 68, 0.1)",
                  color: stat.trend === "up" ? "#10b981" : "#ef4444",
                  fontSize: '0.75rem'
                }}>
                  {stat.trend === "up" ? <ArrowUpRight size={14} /> : <ArrowDownRight size={14} />}
                  {stat.change}
                </div>
              </div>
              <div>
                <h3 className="fw-bold mb-1" style={{ color: 'var(--text-color)', fontSize: '1.75rem' }}>{stat.value}</h3>
                <p className="mb-0 small fw-medium" style={{ color: 'var(--muted-text)' }}>{stat.label}</p>
              </div>
            </div>
          </div>
        ))}
      </div>

      <div className="row g-4 mb-5">
        {/* Revenue Chart */}
        <div className="col-12 col-lg-8">
          <div className="card-custom h-100 p-4">
            <div className="d-flex justify-content-between align-items-center mb-4">
               <div>
                 <h4 className="fw-bold mb-1" style={{ color: 'var(--text-color)' }}>Revenue Overview</h4>
                 <p className="small mb-0" style={{ color: 'var(--muted-text)' }}>Ecommerce vs Ride Hailing performance</p>
               </div>
               <select className="form-select form-select-sm w-auto" style={{
                 backgroundColor: 'var(--bg-color)',
                 borderColor: 'var(--border-color)',
                 color: 'var(--text-color)',
                 borderRadius: '12px',
                 cursor: 'pointer'
               }}>
                 <option>Last 7 Days</option>
                 <option>Last 30 Days</option>
                 <option>This Year</option>
               </select>
            </div>
            
            <div style={{ height: '320px', width: '100%' }}>
              <ResponsiveContainer width="100%" height="100%">
                <AreaChart data={REVENUE_DATA} margin={{ top: 10, right: 10, left: -20, bottom: 0 }}>
                  <defs>
                    <linearGradient id="colorEcommerce" x1="0" y1="0" x2="0" y2="1">
                      <stop offset="5%" stopColor="#f97316" stopOpacity={0.4}/>
                      <stop offset="95%" stopColor="#f97316" stopOpacity={0}/>
                    </linearGradient>
                    <linearGradient id="colorRides" x1="0" y1="0" x2="0" y2="1">
                      <stop offset="5%" stopColor="#3b82f6" stopOpacity={0.4}/>
                      <stop offset="95%" stopColor="#3b82f6" stopOpacity={0}/>
                    </linearGradient>
                  </defs>
                  <CartesianGrid strokeDasharray="3 3" stroke="var(--border-color)" vertical={false} />
                  <XAxis dataKey="name" stroke="var(--muted-text)" fontSize={12} tickLine={false} axisLine={false} dy={10} />
                  <YAxis stroke="var(--muted-text)" fontSize={12} tickLine={false} axisLine={false} tickFormatter={(value) => `$${value}`} />
                  <Tooltip 
                    contentStyle={{ backgroundColor: 'var(--bg-color)', borderColor: 'var(--border-color)', borderRadius: '16px', padding: '12px', color: 'var(--text-color)', boxShadow: 'var(--card-shadow)' }}
                    itemStyle={{ color: 'var(--text-color)', fontSize: '14px', fontWeight: 500 }}
                    labelStyle={{ color: 'var(--muted-text)', marginBottom: '8px' }}
                  />
                  <Area type="monotone" dataKey="ecommerce" stroke="#f97316" strokeWidth={3} fillOpacity={1} fill="url(#colorEcommerce)" activeDot={{ r: 6, strokeWidth: 0 }} />
                  <Area type="monotone" dataKey="rides" stroke="#3b82f6" strokeWidth={3} fillOpacity={1} fill="url(#colorRides)" activeDot={{ r: 6, strokeWidth: 0 }} />
                </AreaChart>
              </ResponsiveContainer>
            </div>
          </div>
        </div>

        {/* User Distribution */}
        <div className="col-12 col-lg-4">
          <div className="card-custom h-100 p-4">
            <div className="mb-4">
              <h4 className="fw-bold mb-1" style={{ color: 'var(--text-color)' }}>User Distribution</h4>
              <p className="small mb-0" style={{ color: 'var(--muted-text)' }}>Platform user breakdown</p>
            </div>
            <div style={{ height: '220px' }} className="d-flex align-items-center justify-content-center mb-4">
              <ResponsiveContainer width="100%" height="100%">
                <PieChart>
                  <Pie
                    data={USER_DISTRIBUTION}
                    cx="50%"
                    cy="50%"
                    innerRadius={65}
                    outerRadius={90}
                    paddingAngle={4}
                    dataKey="value"
                    stroke="none"
                  >
                    {USER_DISTRIBUTION.map((entry, index) => (
                      <Cell key={`cell-${index}`} fill={entry.color} />
                    ))}
                  </Pie>
                  <Tooltip 
                    contentStyle={{ backgroundColor: 'var(--bg-color)', borderColor: 'var(--border-color)', borderRadius: '16px', padding: '12px', boxShadow: 'var(--card-shadow)' }}
                    itemStyle={{ color: 'var(--text-color)', fontWeight: 500 }}
                  />
                </PieChart>
              </ResponsiveContainer>
            </div>
            <div className="pt-3 border-top" style={{ borderColor: 'var(--border-color)' }}>
              {USER_DISTRIBUTION.map((item, i) => (
                <div key={i} className="d-flex align-items-center justify-content-between mb-2 last:mb-0">
                  <div className="d-flex align-items-center gap-2">
                    <div className="rounded-circle" style={{ width: '8px', height: '8px', backgroundColor: item.color }} />
                    <span className="small fw-medium" style={{ color: 'var(--muted-text)' }}>{item.name}</span>
                  </div>
                  <span className="small fw-bold" style={{ color: 'var(--text-color)' }}>{item.value.toLocaleString()}</span>
                </div>
              ))}
            </div>
          </div>
        </div>
      </div>

      {/* Recent Activity */}
      <div className="card-custom p-4">
        <div className="d-flex justify-content-between align-items-center mb-4">
          <div>
            <h4 className="fw-bold mb-1" style={{ color: 'var(--text-color)' }}>Recent Activity</h4>
            <p className="small mb-0" style={{ color: 'var(--muted-text)' }}>Latest platform events</p>
          </div>
          <button className="btn btn-link text-decoration-none fw-semibold p-0" style={{ color: 'var(--primary)' }}>View All →</button>
        </div>
        <div className="row g-4">
          {RECENT_ACTIVITY.map((activity, i) => {
            // Helper to parse color strings from mock data
            const activityColor = activity.color.includes('amber') ? '#f59e0b' : 
                                activity.color.includes('blue') ? '#3b82f6' :
                                activity.color.includes('purple') ? '#8b5cf6' :
                                activity.color.includes('emerald') ? '#10b981' : '#f97316';
            
            return (
              <div key={i} className="col-12 col-md-6 col-lg-4">
                <div className="p-3 rounded-3 d-flex align-items-start gap-3 h-100 transition-all hover:bg-opacity-50" style={{ 
                  backgroundColor: 'var(--hover-bg)', 
                  border: '1px solid var(--border-color)',
                  cursor: 'pointer'
                }}>
                  <div className={`rounded-3 d-flex align-items-center justify-content-center flex-shrink-0`} style={{ 
                    width: '48px', 
                    height: '48px',
                    backgroundColor: `${activityColor}15`, // adding ~10% opacity hex
                    color: activityColor
                  }}>
                    <activity.icon size={20} />
                  </div>
                  <div className="flex-grow-1 min-w-0">
                    <p className="fw-bold mb-1 text-truncate" style={{ color: 'var(--text-color)' }}>{activity.title}</p>
                    <p className="small mb-1 text-truncate" style={{ color: 'var(--muted-text)' }}>{activity.desc}</p>
                    <p className="small opacity-75 mb-0" style={{ color: 'var(--muted-text)', fontSize: '0.75rem' }}>{activity.time}</p>
                  </div>
                </div>
              </div>
            );
          })}
        </div>
      </div>
    </div>
  );
}

