"use client";

import { 
  Clock, 
  MapPin, 
  RefreshCw, 
  Package, 
  Truck, 
  User, 
  AlertCircle,
  MoreVertical,
  ChevronRight,
  Filter,
  Map as MapIcon,
  LayoutGrid
} from "lucide-react";
import { useState, useEffect, useRef } from "react";
import { OrderService } from "@/services/api";
import Link from "next/link";
import PageHeader from "@/components/admin/PageHeader";
import ActionMenu from "@/components/admin/ActionMenu";
import OrderMap from "@/components/admin/OrderMap";

// Status configuration for the monitor
const ACTIVE_STATUSES = ["pending", "preparing", "delivering"];

const STATUS_MAP = {
  pending: { label: "Pending", color: "#f59e0b", bgColor: "rgba(245, 158, 11, 0.1)", icon: Clock },
  preparing: { label: "Preparing", color: "#3b82f6", bgColor: "rgba(59, 130, 246, 0.1)", icon: Package },
  delivering: { label: "Delivering", color: "#a855f7", bgColor: "rgba(168, 85, 247, 0.1)", icon: Truck }
};

export default function ActiveOrdersPage() {
  const [orders, setOrders] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [refreshing, setRefreshing] = useState(false);
  const [lastRefreshed, setLastRefreshed] = useState(new Date());
  const [viewMode, setViewMode] = useState<'grid' | 'map'>('grid');
  const [mapOrders, setMapOrders] = useState<any[]>([]);
  
  // Auto-refresh interval (every 30 seconds)
  const refreshInterval = useRef<NodeJS.Timeout | null>(null);

  useEffect(() => {
    fetchActiveOrders();
    
    // Set up auto-refresh
    refreshInterval.current = setInterval(() => {
      fetchActiveOrders(true);
    }, 30000);

    return () => {
      if (refreshInterval.current) clearInterval(refreshInterval.current);
    };
  }, []);

  const fetchActiveOrders = async (isAuto = false) => {
    if (!isAuto) setLoading(true);
    else setRefreshing(true);
    
    try {
      // We'll fetch multiple statuses and combine, or the API might support status[]
      // For now, let's fetch all orders and filter client-side if API doesn't support multiple statuses at once
      // Actually, standard Laravel filters usually take one or we can do multiple calls
      
      const responses = await Promise.all(
        ACTIVE_STATUSES.map(status => OrderService.getAll({ status, page_size: 50 }))
      );

      const allActiveOrders = responses.flatMap(res => res.data.data || []);
      
      // Sort by newest first
      allActiveOrders.sort((a, b) => new Date(b.created_at).getTime() - new Date(a.created_at).getTime());
      
      setOrders(allActiveOrders);
      
      // Also fetch specialized map locations for the map view
      const mapRes = await OrderService.getActiveLocations();
      setMapOrders(mapRes.data.data || []);
      
      setLastRefreshed(new Date());
    } catch (error) {
      console.error("Failed to fetch active orders:", error);
    } finally {
      setLoading(false);
      setRefreshing(false);
    }
  };

  const getStatusCount = (status: string) => orders.filter(o => o.status === status).length;

  return (
    <div className="py-2">
      <div className="d-flex flex-column flex-md-row justify-content-between align-items-start align-items-md-center gap-3 mb-4">
        <div>
          <PageHeader 
            title="Active Orders Monitor" 
            description="Real-time view of ongoing orders and deliveries."
          />
          <div className="d-flex align-items-center gap-2 mt-1">
             <div className={`rounded-circle ${refreshing ? 'spinner-grow spinner-grow-sm text-success' : 'bg-success'}`} style={{ width: '8px', height: '8px' }}></div>
             <span className="extra-small text-muted-custom">Live Updates Enabled • Last updated: {lastRefreshed.toLocaleTimeString()}</span>
          </div>
        </div>

        <div className="d-flex gap-2">
           <div className="d-flex p-1 rounded-3 bg-white border shadow-sm me-2">
              <button 
                onClick={() => setViewMode('grid')}
                className={`btn btn-sm d-flex align-items-center gap-2 px-3 fw-bold border-0 ${viewMode === 'grid' ? 'btn-primary' : 'text-muted'}`}
              >
                <LayoutGrid size={16} />
                Grid
              </button>
              <button 
                onClick={() => setViewMode('map')}
                className={`btn btn-sm d-flex align-items-center gap-2 px-3 fw-bold border-0 ${viewMode === 'map' ? 'btn-primary' : 'text-muted'}`}
              >
                <MapIcon size={16} />
                Map
              </button>
           </div>

           <button 
             onClick={() => fetchActiveOrders()} 
             disabled={refreshing}
             className="btn btn-white border d-flex align-items-center gap-2 px-3 fw-bold shadow-sm"
           >
             <RefreshCw size={16} className={refreshing ? 'spin-animation' : ''} />
             {refreshing ? 'Refreshing...' : 'Refresh Now'}
           </button>
        </div>
      </div>

      {/* Summary Cards */}
      <div className="row g-3 mb-4">
        {ACTIVE_STATUSES.map(status => {
          const config = STATUS_MAP[status as keyof typeof STATUS_MAP];
          const StatusIcon = config.icon;
          return (
            <div key={status} className="col-12 col-md-4">
              <div className="card-custom p-3 border-0 shadow-sm transition-all hover-translate-y" style={{ borderLeft: `4px solid ${config.color}` }}>
                <div className="d-flex justify-content-between align-items-center mb-2">
                  <div className="rounded-3 p-2" style={{ backgroundColor: config.bgColor, color: config.color }}>
                    <StatusIcon size={20} />
                  </div>
                  <span className="h4 fw-bold mb-0" style={{ color: 'var(--text-color)' }}>{getStatusCount(status)}</span>
                </div>
                <p className="extra-small text-muted-custom text-uppercase fw-bold mb-0" style={{ letterSpacing: '0.5px' }}>{config.label} Orders</p>
              </div>
            </div>
          );
        })}
      </div>

      {/* Monitoring Grid */}
      <div className="card-custom overflow-hidden">
        <div className="px-4 py-3 border-bottom d-flex align-items-center justify-content-between bg-white bg-opacity-50">
           <div className="d-flex align-items-center gap-2">
              <Filter size={16} className="text-muted" />
              <h6 className="fw-bold mb-0">Order Feed</h6>
           </div>
           <span className="badge bg-secondary bg-opacity-10 text-secondary border border-secondary-subtle small fw-bold">{orders.length} Active Total</span>
        </div>

         <div className="p-4">
            {loading ? (
              <div className="text-center py-5">
                 <div className="spinner-border text-primary" role="status"></div>
                 <p className="text-muted-custom mt-2 small">Initializing monitor...</p>
              </div>
            ) : viewMode === 'map' ? (
              <OrderMap orders={mapOrders} />
            ) : orders.length === 0 ? (
              <div className="text-center py-5">
                 <div className="mb-3 d-inline-flex p-4 rounded-circle bg-light border">
                   <Package size={48} className="text-muted opacity-50" />
                 </div>
                 <h5 className="fw-bold" style={{ color: 'var(--text-color)' }}>No active orders at the moment</h5>
                 <p className="text-muted-custom small mb-0">New orders will appear here automatically when they are placed.</p>
              </div>
            ) : (
              <div className="row g-4">
                 {orders.map(order => {
                   const config = STATUS_MAP[order.status as keyof typeof STATUS_MAP] || STATUS_MAP.pending;
                   const StatusIcon = config.icon;
                   
                   return (
                     <div key={order.id} className="col-12 col-xl-6">
                       <div className="p-3 rounded-4 border transition-all hover-shadow-custom bg-white">
                         <div className="d-flex justify-content-between align-items-start mb-3">
                            <div>
                               <div className="d-flex align-items-center gap-2 mb-1">
                                 <h6 className="fw-bold mb-0">Order #{order.id}</h6>
                                 <span 
                                   className="badge rounded-pill small d-flex align-items-center gap-1 border"
                                   style={{
                                     backgroundColor: config.bgColor,
                                     color: config.color,
                                     borderColor: config.color + '22',
                                     fontSize: '0.7rem'
                                   }}
                                 >
                                   <StatusIcon size={10} />
                                   {config.label}
                                 </span>
                               </div>
                               <p className="extra-small text-muted-custom mb-0 d-flex align-items-center gap-1">
                                  <Clock size={12} />
                                  {new Date(order.created_at).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })} • Received {Math.floor((new Date().getTime() - new Date(order.created_at).getTime()) / 60000)}m ago
                               </p>
                            </div>
                            <Link href={`/admin/orders/${order.id}`} className="btn btn-sm btn-light border-0">
                                <ChevronRight size={16} />
                            </Link>
                         </div>

                         <div className="row g-2 mb-3">
                            <div className="col-6">
                               <div className="d-flex align-items-center gap-2">
                                  <User size={14} className="text-muted" />
                                  <span className="small fw-medium text-truncate">{order.user?.name || "Guest"}</span>
                               </div>
                            </div>
                            <div className="col-6 text-end">
                               <span className="small fw-bold text-success">${order.total_amount || order.total}</span>
                            </div>
                            <div className="col-12">
                               <div className="d-flex align-items-center gap-2">
                                  <MapPin size={14} className="text-muted" />
                                  <span className="small text-muted-custom text-truncate">{order.address?.address || order.address || "No address"}</span>
                               </div>
                            </div>
                         </div>

                         <div className="d-flex gap-2">
                            <button 
                              onClick={() => {/* Rapid status transition could go here */}}
                              className="btn btn-sm flex-grow-1 fw-bold border-0 text-white" 
                              style={{ backgroundColor: config.color, opacity: 0.9 }}
                            >
                              Update Workflow
                            </button>
                            <button className="btn btn-sm border fw-bold px-3">
                               Track
                            </button>
                         </div>
                       </div>
                     </div>
                   );
                 })}
              </div>
            )}
         </div>
      </div>

      {/* CSS for animations */}
      <style jsx>{`
        .spin-animation {
          animation: spin 1s linear infinite;
        }
        @keyframes spin {
          from { transform: rotate(0deg); }
          to { transform: rotate(360deg); }
        }
        .hover-translate-y:hover {
          transform: translateY(-4px);
        }
        .hover-shadow-custom:hover {
          box-shadow: 0 10px 20px -5px rgba(0,0,0,0.05);
          border-color: var(--primary-color) !important;
        }
        .extra-small {
          font-size: 0.75rem;
        }
      `}</style>
    </div>
  );
}
