"use client";

import PageHeader from "@/components/admin/PageHeader";
import ActionMenu from "@/components/admin/ActionMenu";
import DeleteModal from "@/components/admin/DeleteModal";
import { Search, MapPin, RefreshCw, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight } from "lucide-react";
import { useState, useEffect } from "react";
import { LocationChargeService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";

export default function AdminLocationChargesPage() {
  const router = useRouter();
  const [locationCharges, setLocationCharges] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [deleteModalOpen, setDeleteModalOpen] = useState(false);
  const [selectedItem, setSelectedItem] = useState<any>(null);
  const [page, setPage] = useState(1);
  const [pagination, setPagination] = useState<any>(null);

  useEffect(() => {
    fetchLocationCharges(page);
  }, [page]);

  const fetchLocationCharges = async (page = 1) => {
    setLoading(true);
    try {
      const params: any = { page };
      if (searchTerm) {
        params.search = searchTerm;
      }
      const response = await LocationChargeService.getAll(params);
      const resData = response.data;
      setLocationCharges(resData.data || []);
      setPagination(resData.meta || resData);
    } catch (error) {
      console.error("Failed to fetch location charges:", error);
      toast.error("Failed to load location charges");
    } finally {
      setLoading(false);
    }
  };

  const handleDeleteClick = (item: any) => {
    setSelectedItem(item);
    setDeleteModalOpen(true);
  };

  const confirmDelete = async () => {
    if (!selectedItem) return;
    try {
      await LocationChargeService.delete(selectedItem.id);
      toast.success("Location charge deleted successfully");
      setDeleteModalOpen(false);
      setSelectedItem(null);
      fetchLocationCharges(page);
    } catch (error) {
      console.error("Failed to delete location charge:", error);
      toast.error("Failed to delete location charge");
    }
  };

  const getTitle = (title: any) => {
    if (!title) return "—";
    if (typeof title === 'string') {
      try {
        const parsed = JSON.parse(title);
        return parsed.en || parsed.ar || title;
      } catch {
        return title;
      }
    }
    if (typeof title === 'object') {
      return title.en || title.ar || "—";
    }
    return String(title);
  };

  return (
    <div className="py-2">
      <PageHeader 
        title="Location Charges" 
        description="Manage location-based surcharges and fees."
        actionLabel="Add New Charge"
        actionLink="/admin/location-charges/create"
      />

      <div className="card-custom overflow-hidden">
        {/* Toolbar */}
        <div className="p-4 border-bottom d-flex flex-column flex-md-row gap-3 justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
          <div className="d-flex p-1 rounded-3" style={{ backgroundColor: 'var(--bg-color)', border: '1px solid var(--border-color)' }}>
             <button
               className="px-3 py-1.5 rounded-2 text-capitalize fw-bold border-0 transition-all text-nowrap shadow-sm"
               style={{
                 fontSize: '0.85rem',
                 backgroundColor: 'var(--text-color)',
                 color: 'var(--bg-color)',
               }}
             >
               All Charges
             </button>
          </div>

          <div className="d-flex gap-2 w-100 w-md-auto">
             <div className="position-relative flex-grow-1" style={{ width: '280px' }}>
                <Search className="position-absolute top-50 translate-middle-y ms-3" size={16} style={{ color: 'var(--muted-text)' }} />
                <input 
                  type="text" 
                  placeholder="Search charges..." 
                  value={searchTerm}
                  onChange={(e) => { setSearchTerm(e.target.value); setPage(1); }}
                  onKeyDown={(e) => { if (e.key === 'Enter') fetchLocationCharges(1); }}
                  className="form-control ps-5 py-2"
                  style={{ 
                    borderRadius: '12px',
                    backgroundColor: 'var(--bg-color)',
                    borderColor: 'var(--border-color)',
                    color: 'var(--text-color)',
                    fontSize: '0.9rem'
                  }}
                />
            </div>
            <button 
                onClick={() => fetchLocationCharges(pagination?.current_page || 1)} 
                className="btn btn-light border d-flex align-items-center justify-content-center"
                style={{ width: '40px', height: '40px', borderColor: 'var(--border-color)' }}
                title="Refresh List"
            >
                <RefreshCw size={18} className={loading ? "animate-spin" : ""} />
            </button>
          </div>
        </div>

        {/* Table */}
        <div className="table-responsive">
          <table className="table mb-0 align-middle">
            <thead style={{ backgroundColor: 'var(--hover-bg)' }}>
              <tr>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Location</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Coordinates</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Radius (km)</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Amount</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom text-end" style={{ borderColor: 'var(--border-color)' }}>Actions</th>
              </tr>
            </thead>
            <tbody>
              {loading ? (
                 <tr>
                    <td colSpan={5} className="text-center py-5">
                       <div className="spinner-border text-primary" role="status">
                           <span className="visually-hidden">Loading...</span>
                       </div>
                    </td>
                 </tr>
              ) : locationCharges.length === 0 ? (
                <tr>
                   <td colSpan={5} className="text-center py-5 text-muted">
                      No location charges found.
                   </td>
                </tr>
              ) : (
                locationCharges.map((item) => (
                  <tr key={item.id} style={{ borderBottom: '1px solid var(--border-color)' }}>
                    <td className="px-4 py-3">
                      <div className="d-flex align-items-center gap-3">
                        <div className="rounded-3 d-flex align-items-center justify-content-center" style={{ 
                            width: '40px', 
                            height: '40px', 
                            backgroundColor: 'var(--hover-bg)',
                            color: 'var(--primary-color)',
                            border: '1px solid var(--border-color)'
                        }}>
                          <MapPin size={20} />
                        </div>
                        <div>
                          <p className="fw-bold mb-0" style={{ color: 'var(--text-color)' }}>{getTitle(item.title)}</p>
                          <p className="small mb-0 text-muted-custom">ID: #{item.id}</p>
                        </div>
                      </div>
                    </td>
                    <td className="px-4 py-3">
                       <span className="small text-muted-custom font-monospace">
                         {Number(item.latitude || 0).toFixed(4)}, {Number(item.longitude || 0).toFixed(4)}
                       </span>
                    </td>
                    <td className="px-4 py-3">
                       <span className="badge bg-info bg-opacity-10 text-info border border-info-subtle">
                         {Number(item.radius || 0).toFixed(2)} km
                       </span>
                    </td>
                     <td className="px-4 py-3">
                       <span className="fw-bold text-success">${Number(item.amount || 0).toFixed(2)}</span>
                    </td>
                    <td className="px-4 py-3 text-end">
                      <ActionMenu 
                        onView={() => router.push(`/admin/location-charges/${item.id}`)}
                        onEdit={() => router.push(`/admin/location-charges/${item.id}/edit`)}
                        onDelete={() => handleDeleteClick(item)}
                      />
                    </td>
                  </tr>
                ))
              )}
            </tbody>
          </table>
        </div>
      </div>

      {/* Pagination */}
      {pagination && pagination.total > 0 && (
         <div className="d-flex justify-content-between align-items-center mt-4">
           <div className="small text-muted-custom">
             Showing <span className="fw-bold text-dark">{pagination.from || 0}</span> to <span className="fw-bold text-dark">{pagination.to || 0}</span> of <span className="fw-bold text-dark">{pagination.total || 0}</span> results
           </div>
           <div className="d-flex gap-2 flex-wrap">
             <button
               onClick={() => fetchLocationCharges(1)}
               disabled={pagination.current_page === 1}
               className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
               style={{ padding: '0.5rem 1rem' }}
             >
               <ChevronsLeft size={16} /> First
             </button>
             <button
               onClick={() => fetchLocationCharges(pagination.current_page - 1)}
               disabled={pagination.current_page === 1}
               className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
               style={{ padding: '0.5rem 1rem' }}
             >
               <ChevronLeft size={16} /> Previous
             </button>
             
             <div className="d-flex align-items-center gap-1 flex-wrap">
               {pagination.links
                 ?.filter((link: any) => {
                   const pageNum = parseInt(link.label);
                   return !isNaN(pageNum) && link.label.trim() === pageNum.toString();
                 })
                 .map((link: any, index: number) => {
                   const isCurrent = link.active;
                   const pageNumber = parseInt(link.label);
                   
                   return (
                     <button
                       key={index}
                       onClick={() => fetchLocationCharges(pageNumber)}
                       disabled={isCurrent}
                       className={`btn btn-sm ${isCurrent ? 'btn-primary' : 'btn-light border'}`}
                       style={{ minWidth: '32px', height: '32px', padding: '0 0.5rem' }}
                     >
                       {link.label}
                     </button>
                   );
                 })}
             </div>

             <button
               onClick={() => fetchLocationCharges(pagination.current_page + 1)}
               disabled={pagination.current_page === pagination.last_page}
               className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
               style={{ padding: '0.5rem 1rem' }}
             >
               Next <ChevronRight size={16} />
             </button>
             <button
               onClick={() => fetchLocationCharges(pagination.last_page)}
               disabled={pagination.current_page === pagination.last_page}
               className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
               style={{ padding: '0.5rem 1rem' }}
             >
               Last <ChevronsRight size={16} />
             </button>
           </div>
         </div>
      )}

      <DeleteModal 
        isOpen={deleteModalOpen}
        onClose={() => setDeleteModalOpen(false)}
        onConfirm={confirmDelete}
        itemName={getTitle(selectedItem?.title) || undefined}
        title="Delete Location Charge"
        description="Are you sure you want to delete this location charge? This action cannot be undone."
      />
    </div>
  );
}
