"use client";

import PageHeader from "@/components/admin/PageHeader";
import ActionMenu from "@/components/admin/ActionMenu";
import DeleteModal from "@/components/admin/DeleteModal";
import { Search, Ticket, Calendar, Tag, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight, Loader2, Plus, RefreshCw } from "lucide-react";
import { useState, useEffect } from "react";
import { CouponService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";

export default function AdminCouponsPage() {
  const router = useRouter();
  const [coupons, setCoupons] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [page, setPage] = useState(1);
  const [pagination, setPagination] = useState<any>(null);
  const [deleteModalOpen, setDeleteModalOpen] = useState(false);
  const [selectedCoupon, setSelectedCoupon] = useState<any>(null);
  const [mounted, setMounted] = useState(false);

  useEffect(() => {
    setMounted(true);
    fetchCoupons();
  }, [page, searchTerm]);

  const fetchCoupons = async () => {
    setLoading(true);
    try {
      const params: any = { page };
      if (searchTerm) {
        params.code = searchTerm;
      }
      const response = await CouponService.getAll(params);
      const resData = response.data;
      setCoupons(resData.data || []);
      setPagination(resData.meta || resData);
    } catch (error) {
      console.error("Failed to fetch coupons:", error);
      toast.error("Failed to load coupons");
    } finally {
      setLoading(false);
    }
  };

  const handleDeleteClick = (coupon: any) => {
    setSelectedCoupon(coupon);
    setDeleteModalOpen(true);
  };

  const confirmDelete = async () => {
    if (!selectedCoupon) return;
    try {
      await CouponService.delete(selectedCoupon.id);
      toast.success("Coupon deleted successfully");
      setDeleteModalOpen(false);
      setSelectedCoupon(null);
      fetchCoupons();
    } catch (error) {
      console.error("Failed to delete coupon:", error);
      toast.error("Failed to delete coupon");
    }
  };

  const formatDate = (dateString: string | null | undefined) => {
    if (!dateString) return "—";
    const date = new Date(dateString);
    if (isNaN(date.getTime())) return "—";
    return date.toLocaleDateString();
  };

  const getLocalizedValue = (value: any) => {
    if (!value) return "—";
    if (typeof value === 'string') {
      try {
        const parsed = JSON.parse(value);
        if (typeof parsed === 'object') return parsed.en || parsed.ar || Object.values(parsed)[0] || "—";
      } catch (e) {
        return value;
      }
    }
    if (typeof value === 'object') {
      return value.en || value.ar || Object.values(value)[0] || "—";
    }
    return String(value);
  };

  return (
    <div className="py-2">
      <PageHeader 
        title="Coupon Management" 
        description="Manage discounts, promotional codes, and rewards."
        actionLabel="Add New Coupon"
        actionLink="/admin/coupons/create"
      />

      <div className="card-custom overflow-hidden">
        {/* Toolbar */}
        <div className="p-4 border-bottom d-flex flex-column flex-md-row gap-3 justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
          <div className="position-relative" style={{ width: '320px' }}>
            <Search className="position-absolute top-50 translate-middle-y ms-3" size={16} style={{ color: 'var(--muted-text)' }} />
            <input 
              type="text" 
              placeholder="Search by code..." 
              value={searchTerm}
              onChange={(e) => { setSearchTerm(e.target.value); setPage(1); }}
              className="form-control ps-5 py-2"
              style={{ 
                borderRadius: '12px',
                backgroundColor: 'var(--bg-color)',
                borderColor: 'var(--border-color)',
                color: 'var(--text-color)',
                fontSize: '0.9rem'
              }}
            />
          </div>

          <button 
            onClick={() => fetchCoupons()} 
            className="btn btn-light border d-flex align-items-center justify-content-center"
            style={{ width: '40px', height: '40px', borderColor: 'var(--border-color)' }}
            title="Refresh List"
          >
            <RefreshCw size={18} className={loading ? "spin-animation" : ""} />
          </button>
        </div>

        {/* Table */}
        <div className="table-responsive">
          {loading ? (
             <div className="d-flex flex-column align-items-center justify-content-center py-5 w-100">
                <Loader2 className="spin-animation text-primary mb-2" size={32} />
                <p className="text-muted-custom small">Loading coupons...</p>
             </div>
          ) : coupons.length === 0 ? (
            <div className="d-flex flex-column align-items-center justify-content-center py-5 opacity-50 w-100">
               <Ticket size={48} className="mb-2" />
               <p>No coupons found.</p>
            </div>
          ) : (
            <>
              <table className="table mb-0 align-middle">
                <thead style={{ backgroundColor: 'var(--hover-bg)' }}>
                  <tr>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Code & Title</th>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Type</th>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Reward</th>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Expires</th>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Status</th>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom text-end" style={{ borderColor: 'var(--border-color)' }}>Actions</th>
                  </tr>
                </thead>
                <tbody>
                  {coupons.map((coupon) => (
                    <tr key={coupon.id} style={{ borderBottom: '1px solid var(--border-color)' }}>
                      <td className="px-4 py-3">
                        <div className="d-flex align-items-center gap-3">
                          <div className="p-2 rounded-3 bg-primary bg-opacity-10 text-primary">
                            <Tag size={20} />
                          </div>
                          <div>
                            <p className="fw-bold mb-0" style={{ color: 'var(--text-color)' }}>{coupon.code}</p>
                            <p className="extra-small text-muted-custom mb-0">{getLocalizedValue(coupon.title)}</p>
                          </div>
                        </div>
                      </td>
                      <td className="px-4 py-3">
                        <span className="small fw-medium text-capitalize">{coupon.type?.replace('_', ' ') || "—"}</span>
                      </td>
                      <td className="px-4 py-3">
                        <span className="fw-bold text-success">
                          {coupon.type === 'percent' || coupon.type === 'percentage' ? `${coupon.reward}%` : `$${Number(coupon.reward || 0).toFixed(2)}`}
                        </span>
                      </td>
                      <td className="px-4 py-3">
                        <div className="d-flex align-items-center gap-2 small text-muted-custom">
                          <Calendar size={14} />
                          {mounted ? formatDate(coupon.expires_at) : "—"}
                        </div>
                      </td>
                      <td className="px-4 py-3">
                        {mounted && new Date(coupon.expires_at) < new Date() ? (
                          <span className="badge rounded-pill bg-danger-subtle text-danger border border-danger-subtle px-3 py-1.5 extra-small fw-bold text-uppercase">Expired</span>
                        ) : (
                          <span className="badge rounded-pill bg-success-subtle text-success border border-success-subtle px-3 py-1.5 extra-small fw-bold text-uppercase">Active</span>
                        )}
                      </td>
                      <td className="px-4 py-3 text-end">
                        <ActionMenu 
                          onView={() => router.push(`/admin/coupons/${coupon.id}`)}
                          onEdit={() => router.push(`/admin/coupons/${coupon.id}/edit`)}
                          onDelete={() => handleDeleteClick(coupon)}
                        />
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>

              {/* Pagination */}
              {pagination && pagination.last_page > 1 && (
                <div className="p-4 border-top d-flex justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
                  <div className="small text-muted-custom">
                    Showing <span className="fw-bold text-dark">{pagination.from || 0}</span> to <span className="fw-bold text-dark">{pagination.to || 0}</span> of <span className="fw-bold text-dark">{pagination.total || 0}</span> results
                  </div>
                  <div className="d-flex gap-2 flex-wrap">
                    <button
                      onClick={() => setPage(1)}
                      disabled={pagination.current_page === 1}
                      className="btn btn-white border btn-sm d-flex align-items-center justify-content-center p-2 rounded-3 shadow-sm transition-all"
                    >
                      <ChevronsLeft size={16} />
                    </button>
                    <button
                      onClick={() => setPage(prev => Math.max(prev - 1, 1))}
                      disabled={pagination.current_page === 1}
                      className="btn btn-white border btn-sm d-flex align-items-center justify-content-center p-2 rounded-3 shadow-sm transition-all"
                    >
                      <ChevronLeft size={16} />
                    </button>
                    
                    <div className="d-flex align-items-center gap-1 flex-wrap">
                      {pagination.links
                        ?.filter((link: any) => {
                          const pageNum = parseInt(link.label);
                          return !isNaN(pageNum) && link.label.trim() === pageNum.toString();
                        })
                        .map((link: any, index: number) => {
                          const isCurrent = link.active;
                          const pageNumber = parseInt(link.label);
                          return (
                            <button
                              key={index}
                              onClick={() => setPage(pageNumber)}
                              disabled={isCurrent}
                              className={`btn btn-sm ${isCurrent ? 'btn-primary' : 'btn-light border'}`}
                              style={{ minWidth: '32px', height: '32px', padding: '0 0.5rem' }}
                            >
                              {link.label}
                            </button>
                          );
                        })}
                    </div>
                    
                    <button
                      onClick={() => setPage(prev => Math.min(prev + 1, pagination.last_page))}
                      disabled={pagination.current_page === pagination.last_page}
                      className="btn btn-white border btn-sm d-flex align-items-center justify-content-center p-2 rounded-3 shadow-sm transition-all"
                    >
                      <ChevronRight size={16} />
                    </button>
                    <button
                      onClick={() => setPage(pagination.last_page)}
                      disabled={pagination.current_page === pagination.last_page}
                      className="btn btn-white border btn-sm d-flex align-items-center justify-content-center p-2 rounded-3 shadow-sm transition-all"
                    >
                      <ChevronsRight size={16} />
                    </button>
                  </div>
                </div>
              )}
            </>
          )}
        </div>
      </div>

      <DeleteModal 
        isOpen={deleteModalOpen}
        onClose={() => setDeleteModalOpen(false)}
        onConfirm={confirmDelete}
        itemName={selectedCoupon?.code}
        title="Delete Coupon"
        description="Are you sure you want to delete this coupon? This action cannot be undone."
      />
    </div>
  );
}
