"use client";

import PageHeader from "@/components/admin/PageHeader";
import { 
  Plus, 
  Trash2, 
  ArrowLeft, 
  Ticket, 
  Calendar, 
  Percent, 
  DollarSign, 
  Languages, 
  Globe, 
  Info,
  Settings,
  ChevronRight,
  CheckCircle2,
  X,
  Loader2,
  Upload
} from "lucide-react";
import { useState, useEffect } from "react";
import { CouponService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";
import Link from "next/link";

interface MetaField {
  key: string;
  value: string;
}

export default function AdminCreateCouponPage() {
  const router = useRouter();
  const [loading, setLoading] = useState(false);
  const [mounted, setMounted] = useState(false);

  useEffect(() => {
    setMounted(true);
  }, []);
  
  // Basic Fields
  const [formData, setFormData] = useState({
    code: "",
    type: "fixed",
    reward: "",
    expires_at: "",
    is_applies_to_all: true,
    image: null as File | null,
  });

  const [imagePreview, setImagePreview] = useState<string | null>(null);

  const handleImageChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      setFormData(prev => ({ ...prev, image: file }));
      setImagePreview(URL.createObjectURL(file));
    }
  };

  // Translations
  const [translations, setTranslations] = useState({
    title_en: "",
    title_ar: "",
    detail_en: "",
    detail_ar: "",
  });

  // Meta Fields
  const [metaFields, setMetaFields] = useState<MetaField[]>([]);

  // Tab State
  const [activeTab, setActiveTab] = useState<'english' | 'arabic'>('english');

  const handleAddMetaField = () => {
    setMetaFields([...metaFields, { key: "", value: "" }]);
  };

  const handleRemoveMetaField = (index: number) => {
    setMetaFields(metaFields.filter((_, i) => i !== index));
  };

  const handleMetaFieldChange = (index: number, field: keyof MetaField, value: string) => {
    const newFields = [...metaFields];
    newFields[index][field] = value;
    setMetaFields(newFields);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      // Format translations for backend validation (array format)
      const title_translations_array = [];
      if (translations.title_en) title_translations_array.push({ language: "en", text: translations.title_en });
      if (translations.title_ar) title_translations_array.push({ language: "ar", text: translations.title_ar });

      const detail_translations_array = [];
      if (translations.detail_en) detail_translations_array.push({ language: "en", text: translations.detail_en });
      if (translations.detail_ar) detail_translations_array.push({ language: "ar", text: translations.detail_ar });

      // Format translations for controller (object format)
      const title_translations_obj = {
        en: translations.title_en,
        ar: translations.title_ar
      };

      const detail_translations_obj = {
        en: translations.detail_en,
        ar: translations.detail_ar
      };

      // Format meta fields
      const meta: Record<string, string> = {};
      metaFields.forEach(field => {
        if (field.key) meta[field.key] = field.value;
      });

      const payloadData: any = {
        ...formData,
        applies_to_all: formData.is_applies_to_all,
        // For validation
        title_translations: title_translations_array,
        detail_translations: detail_translations_array,
        // For controller
        _title: title_translations_obj,
        _detail: detail_translations_obj,
        meta: JSON.stringify(meta),
      };

      if (formData.image) {
        const formDataPayload = new FormData();
        Object.entries(payloadData).forEach(([key, value]) => {
          if (key === 'image') return;
          if (typeof value === 'object' && value !== null) {
            formDataPayload.append(key, JSON.stringify(value));
          } else {
            formDataPayload.append(key, String(value));
          }
        });
        formDataPayload.append('image', formData.image);
        await CouponService.create(formDataPayload);
      } else {
        await CouponService.create(payloadData);
      }
      toast.success("Coupon created successfully");
      router.push("/admin/coupons");
    } catch (error: any) {
      console.error("Failed to create coupon:", error);
      toast.error(error.response?.data?.message || "Failed to create coupon");
    } finally {
      setLoading(false);
    }
  };

  if (!mounted) return null;

  return (
    <div className="py-2">
      <div className="mb-4">
        <Link 
          href="/admin/coupons" 
          className="btn btn-link text-muted-custom p-0 d-flex align-items-center gap-2 mb-3 text-decoration-none hover-text-primary transition-all"
        >
          <ArrowLeft size={16} />
          <span>Back to Coupons</span>
        </Link>
        <PageHeader 
          title="Create New Coupon" 
          description="Design a new promotional offer or discount code."
        />
      </div>

      <form onSubmit={handleSubmit} className="row g-4">
        <div className="col-12 col-lg-8">
          {/* Coupon Identity */}
          <div className="card-custom p-4 mb-4">
            <h5 className="fw-bold mb-4 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
              <Ticket size={20} className="text-primary-custom" />
              Coupon Identity
            </h5>
            
            <div className="row g-3">
              <div className="col-md-6">
                <label className="form-label small fw-bold text-muted-custom text-uppercase">Coupon Code</label>
                <input 
                  type="text" 
                  className="form-control" 
                  placeholder="e.g. SUMMER50"
                  required
                  value={formData.code}
                  onChange={(e) => setFormData({...formData, code: e.target.value.toUpperCase()})}
                />
                <div className="form-text extra-small">Uppercase letters and numbers only.</div>
              </div>
              
              <div className="col-md-6">
                <label className="form-label small fw-bold text-muted-custom text-uppercase">Applies To All</label>
                <div className="form-check form-switch p-0 mt-1 d-flex align-items-center justify-content-between border rounded-3 px-3 py-2">
                   <span className="small text-muted-custom">Available for all users</span>
                   <input 
                    className="form-check-input mt-0 ms-0" 
                    type="checkbox" 
                    id="appliesToAll"
                    checked={formData.is_applies_to_all}
                    onChange={(e) => setFormData({...formData, is_applies_to_all: e.target.checked})}
                   />
                </div>
              </div>

              <div className="col-12 pt-2">
                <div className="p-3 rounded-4 bg-light bg-opacity-10 border border-dashed">
                  <div className="row g-3">
                    <div className="col-md-6">
                      <label className="form-label small fw-bold text-muted-custom text-uppercase">Reward Type</label>
                      <select 
                        className="form-select"
                        value={formData.type}
                        onChange={(e) => setFormData({...formData, type: e.target.value})}
                      >
                        <option value="fixed">Fixed Amount ($)</option>
                        <option value="percent">Percentage (%)</option>
                      </select>
                    </div>
                    <div className="col-md-6">
                      <label className="form-label small fw-bold text-muted-custom text-uppercase">Reward Value</label>
                      <div className="input-group">
                        <span className="input-group-text bg-white border-end-0">
                          {formData.type === 'fixed' ? <DollarSign size={16} /> : <Percent size={16} />}
                        </span>
                        <input 
                          type="number" 
                          className="form-control border-start-0 ps-0" 
                          placeholder="0.00"
                          step="0.01"
                          required
                          value={formData.reward}
                          onChange={(e) => setFormData({...formData, reward: e.target.value})}
                        />
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Translations */}
          <div className="card-custom p-4 mb-4">
             <h5 className="fw-bold mb-4 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
              <Languages size={20} className="text-primary-custom" />
              Content & Localization
            </h5>
            
            <ul className="nav nav-tabs border-0 mb-4 bg-light bg-opacity-10 p-1 rounded-3" role="tablist">
              <li className="nav-item flex-grow-1" role="presentation">
                <button 
                  className={`nav-link w-100 border-0 rounded-2 fw-bold small py-2 d-flex align-items-center justify-content-center gap-2 ${activeTab === 'english' ? 'active' : ''}`}
                  onClick={() => setActiveTab('english')}
                  type="button"
                >
                   <Globe size={14} /> English
                </button>
              </li>
              <li className="nav-item flex-grow-1" role="presentation">
                <button 
                  className={`nav-link w-100 border-0 rounded-2 fw-bold small py-2 d-flex align-items-center justify-content-center gap-2 ${activeTab === 'arabic' ? 'active' : ''}`}
                  onClick={() => setActiveTab('arabic')}
                  type="button" 
                  dir="rtl"
                >
                   العربية <Globe size={14} />
                </button>
              </li>
            </ul>

            <div className="tab-content">
              <div className={`tab-pane fade ${activeTab === 'english' ? 'show active' : ''}`} role="tabpanel">
                <div className="mb-3">
                  <label className="form-label small fw-bold text-muted-custom text-uppercase">Title (English)</label>
                  <input 
                    type="text" 
                    className="form-control" 
                    placeholder="Enter coupon title"
                    value={translations.title_en}
                    onChange={(e) => setTranslations({...translations, title_en: e.target.value})}
                  />
                </div>
                <div className="mb-0">
                  <label className="form-label small fw-bold text-muted-custom text-uppercase">Detail (English)</label>
                  <textarea 
                    className="form-control" 
                    rows={3} 
                    placeholder="Enter coupon description"
                    value={translations.detail_en}
                    onChange={(e) => setTranslations({...translations, detail_en: e.target.value})}
                  ></textarea>
                </div>
              </div>
              <div className={`tab-pane fade ${activeTab === 'arabic' ? 'show active' : ''}`} role="tabpanel">
                <div className="mb-3 text-end" dir="rtl">
                  <label className="form-label small fw-bold text-muted-custom text-uppercase">العنوان (العربية)</label>
                  <input 
                    type="text" 
                    className="form-control text-end" 
                    placeholder="أدخل عنوان الكوبون"
                    value={translations.title_ar}
                    onChange={(e) => setTranslations({...translations, title_ar: e.target.value})}
                  />
                </div>
                <div className="mb-0 text-end" dir="rtl">
                  <label className="form-label small fw-bold text-muted-custom text-uppercase">التفاصيل (العربية)</label>
                  <textarea 
                    className="form-control text-end" 
                    rows={3} 
                    placeholder="أدخل وصف الكوبون"
                    value={translations.detail_ar}
                    onChange={(e) => setTranslations({...translations, detail_ar: e.target.value})}
                  ></textarea>
                </div>
              </div>
            </div>
          </div>
        </div>

        <div className="col-12 col-lg-4">
          {/* Scheduling */}
          <div className="card-custom p-4 mb-4">
             <h5 className="fw-bold mb-4 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
              <Calendar size={20} className="text-primary-custom" />
              Scheduling
            </h5>
            <div className="mb-0">
              <label className="form-label small fw-bold text-muted-custom text-uppercase">Expiry Date</label>
              <input 
                type="date" 
                className="form-control" 
                required
                value={formData.expires_at}
                onChange={(e) => setFormData({...formData, expires_at: e.target.value})}
              />
              <div className="form-text extra-small mt-2">The coupon will be inactive after this date.</div>
            </div>
          </div>

          {/* Meta Fields */}
          <div className="card-custom p-4 mb-4">
             <h5 className="fw-bold mb-4 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
              <Settings size={20} className="text-primary-custom" />
              Meta Fields
            </h5>
            
            <div className="d-flex flex-column gap-3 mb-4">
              {metaFields.map((field, index) => (
                <div key={index} className="p-3 rounded-4 border bg-light bg-opacity-10 position-relative group">
                   <button 
                    type="button"
                    onClick={() => handleRemoveMetaField(index)}
                    className="position-absolute top-0 end-0 m-2 btn btn-link text-danger p-0"
                   >
                    <X size={14} />
                   </button>
                   <div className="mb-2">
                    <label className="extra-small fw-bold text-muted-custom text-uppercase">Key</label>
                    <input 
                      type="text" 
                      className="form-control form-control-sm border-dashed" 
                      placeholder="key"
                      value={field.key}
                      onChange={(e) => handleMetaFieldChange(index, 'key', e.target.value)}
                    />
                   </div>
                   <div>
                    <label className="extra-small fw-bold text-muted-custom text-uppercase">Value</label>
                    <input 
                      type="text" 
                      className="form-control form-control-sm border-dashed" 
                      placeholder="value"
                      value={field.value}
                      onChange={(e) => handleMetaFieldChange(index, 'value', e.target.value)}
                    />
                   </div>
                </div>
              ))}
              
              <button 
                type="button" 
                onClick={handleAddMetaField}
                className="btn btn-outline-primary btn-sm d-flex align-items-center justify-content-center gap-2 py-2 rounded-3"
                style={{ borderStyle: 'dashed' }}
              >
                <Plus size={16} />
                <span>Add Meta Field</span>
              </button>
            </div>
          </div>

          {/* Image Upload */}
          <div className="card-custom p-4 mb-4">
            <h5 className="fw-bold mb-4 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
              <Globe size={20} className="text-primary-custom" />
              Coupon Image
            </h5>
            
            <div className="d-flex flex-column gap-2">
              <div className="d-flex align-items-center justify-content-center p-3 rounded-3 border-2 border-dashed transition-all hover-bg-custom position-relative overflow-hidden" 
                style={{ 
                  borderColor: 'var(--border-color)', 
                  backgroundColor: 'var(--bg-opacity-5)',
                  cursor: 'pointer',
                  minHeight: '200px'
                }}
              >
                <input 
                   type="file" 
                   accept="image/*"
                   onChange={handleImageChange}
                   className="position-absolute w-100 h-100 opacity-0 cursor-pointer"
                   style={{ zIndex: 10 }}
                />
                {imagePreview ? (
                    <div className="w-100 h-100 position-absolute top-0 start-0">
                        <img src={imagePreview} alt="Preview" className="w-100 h-100 object-fit-cover" />
                        <div className="position-absolute bottom-0 start-0 w-100 p-2 bg-dark bg-opacity-50 text-white text-center small">
                            Click to change
                        </div>
                    </div>
                ) : (
                  <div className="text-center">
                      <div className="mb-3 d-inline-flex p-3 rounded-circle shadow-sm" style={{ backgroundColor: 'var(--hover-bg)' }}>
                        <Upload size={24} className="text-primary-custom" />
                      </div>
                      <p className="small fw-bold mb-1" style={{ color: 'var(--text-color)' }}>Choose Banner</p>
                      <p className="small text-muted-custom mb-0">Recommended: 800x400px</p>
                  </div>
                )}
              </div>
            </div>
          </div>

          {/* Action Card */}
          <div className="card-custom p-4 bg-primary bg-opacity-10 border-primary border-opacity-10 shadow-sm text-center">
             <div className="rounded-circle bg-white shadow-sm p-3 mx-auto mb-3" style={{ width: 'fit-content' }}>
                <Ticket size={24} className="text-primary-custom" />
             </div>
             <h6 className="fw-bold mb-2">Ready to publish?</h6>
             <p className="extra-small text-muted-custom mb-3 px-3">Double-check the expiry date and reward value before saving.</p>
             <button 
                type="submit" 
                className="btn btn-primary w-100 py-2.5 rounded-3 d-flex align-items-center justify-content-center gap-2"
                disabled={loading}
             >
                {loading ? <Loader2 size={18} className="spin-animation" /> : <><Plus size={18} /> Create Coupon</>}
             </button>
          </div>
        </div>
      </form>
    </div>
  );
}
