"use client";

import PageHeader from "@/components/admin/PageHeader";
import { Save, X, Upload, Globe, Plus, Trash2, FileText, Layers, Tag, Image as ImageIcon, MoreHorizontal } from "lucide-react";
import Link from "next/link";
import { useState, useEffect } from "react";
import { CategoryService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";

interface MetaField {
  key: string;
  value: string;
}

export default function CreateCategoryPage() {
  const router = useRouter();
  const [loading, setLoading] = useState(false);
  const [categories, setCategories] = useState<any[]>([]);
  const [imagePreview, setImagePreview] = useState<string | null>(null);
  const [bannerPreview, setBannerPreview] = useState<string | null>(null);

  // Form State
  const [formData, setFormData] = useState({
    title_en: "",
    title_ar: "",
    slug: "",
    sort_order: "1",
    scope: "ecommerce",
    parent_id: "",
    image: null as File | null,
    banner: null as File | null
  });

  const [metaFields, setMetaFields] = useState<MetaField[]>([
    { key: "scope", value: "ecommerce" },
    { key: "vendor_type", value: "" }
  ]);

  const [manualSlug, setManualSlug] = useState(false);

  useEffect(() => {
    fetchCategories();
  }, []);

  // Auto-generate slug from title_en
  useEffect(() => {
    if (!manualSlug) {
      const generatedSlug = (formData.title_en || '')
        .toLowerCase()
        .replace(/[^a-z0-9\s-]/g, '') // Remove non-alphanumeric except spaces/hyphens
        .replace(/\s+/g, '-')          // Replace spaces with -
        .replace(/--+/g, '-')          // Replace multiple - with single -
        .replace(/^-+|-+$/g, '')      // Trim leading/trailing -
        .trim();
      setFormData(prev => ({ ...prev, slug: generatedSlug }));
    }
  }, [formData.title_en, manualSlug]);

  const fetchCategories = async () => {
    try {
      const response = await CategoryService.getAll();
      setCategories(response.data.data || response.data || []);
    } catch (err) {
      console.error("Failed to load categories", err);
    }
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value } = e.target;
    if (name === 'slug') setManualSlug(true);
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  const handleImageChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      setFormData(prev => ({ ...prev, image: file }));
      setImagePreview(URL.createObjectURL(file));
    }
  };

  const handleBannerChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      setFormData(prev => ({ ...prev, banner: file }));
      setBannerPreview(URL.createObjectURL(file));
    }
  };

  // Meta Fields Logic
  const handleMetaChange = (index: number, field: 'key' | 'value', value: string) => {
    const newMeta = [...metaFields];
    newMeta[index][field] = value;
    setMetaFields(newMeta);
  };
  const addMetaField = () => setMetaFields([...metaFields, { key: "", value: "" }]);
  const removeMetaField = (index: number) => setMetaFields(metaFields.filter((_, i) => i !== index));

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      const data = new FormData();
      
      // Basic info
      data.append('name', formData.title_en);
      data.append('slug', formData.slug);
      data.append('sort_order', formData.sort_order);
      data.append('active', '1');
      if (formData.parent_id) {
        data.append('parent_id', formData.parent_id);
      }

      // Translations - must be JSON string as per backend validator
      data.append('title_translations', JSON.stringify([
        { language: 'en', text: formData.title_en },
        { language: 'ar', text: formData.title_ar }
      ]));

      // Meta fields
      const meta = metaFields.reduce((acc, field) => {
        if (field.key) acc[field.key] = field.value;
        return acc;
      }, {} as Record<string, string>);
      data.append('meta', JSON.stringify(meta));

      // Images - can be sent in the same request
      if (formData.image) {
        data.append('image', formData.image);
      }
      if (formData.banner) {
        data.append('banner', formData.banner);
      }

      await CategoryService.create(data);

      toast.success("Category created successfully!");
      router.push("/admin/categories");

    } catch (err: any) {
      console.error("Creation error:", err);
      
      // Handle detailed validation errors
      if (err.response?.data?.errors) {
        const errors = err.response.data.errors;
        Object.keys(errors).forEach(key => {
          const messages = errors[key];
          if (Array.isArray(messages)) {
            messages.forEach(msg => toast.error(`${key}: ${msg}`));
          } else {
            toast.error(`${key}: ${messages}`);
          }
        });
      } else {
        toast.error(err.response?.data?.message || "Failed to create category");
      }
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="py-2">
      <PageHeader title="Create Category" description="Add a new product category." />

      <div className="row justify-content-center">
        <div className="col-12">
          <form onSubmit={handleSubmit}>
            <div className="row g-4">
              
              {/* LEFT COLUMN: Main Content */}
              <div className="col-12 col-lg-8">
                
                {/* General Information */}
                <div className="card-custom p-3 mb-3">
                  <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                    <FileText size={20} className="text-primary-custom" />
                    General Information
                  </h5>
                  
                  {/* Title Section */}
                  <div className="mb-4">
                     <h6 className="small fw-bold text-muted mb-2 border-bottom pb-1">Category Title</h6>
                     <div className="row g-3">
                      <div className="col-12 col-md-6">
                        <label className="form-label small fw-bold text-muted-custom">Title (English)</label>
                        <input 
                          type="text" 
                          name="title_en" 
                          value={formData.title_en} 
                          onChange={handleChange} 
                          className="form-control" 
                          placeholder="Category Name" 
                          required 
                          style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                        />
                      </div>
                      <div className="col-12 col-md-6">
                        <label className="form-label small fw-bold text-muted-custom">Title (Arabic)</label>
                        <input 
                          type="text" 
                          name="title_ar" 
                          value={formData.title_ar} 
                          onChange={handleChange} 
                          className="form-control" 
                          placeholder="اسم الفئة" 
                          required 
                          style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                        />
                      </div>
                    </div>
                  </div>

                  {/* Slug & Sort Order */}
                  <div>
                     <h6 className="small fw-bold text-muted mb-2 border-bottom pb-1">URL & Ordering</h6>
                     <div className="row g-3">
                      <div className="col-12 col-md-8">
                        <label className="form-label small fw-bold text-muted-custom">Slug</label>
                        <input 
                          type="text" 
                          name="slug" 
                          value={formData.slug} 
                          onChange={handleChange} 
                          className="form-control" 
                          placeholder="category-slug" 
                          required 
                          style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                        />
                        <small className="text-muted d-block mt-1" style={{ fontSize: '0.7rem' }}>URL-friendly name (e.g., burger-food)</small>
                      </div>
                      <div className="col-12 col-md-4">
                        <label className="form-label small fw-bold text-muted-custom">Sort Order</label>
                        <input 
                          type="number" 
                          name="sort_order" 
                          value={formData.sort_order} 
                          onChange={handleChange} 
                          className="form-control" 
                          min="1"
                          style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                        />
                      </div>
                    </div>
                  </div>
                </div>

                {/* Meta Fields */}
                <div className="card-custom p-3">
                  <div className="d-flex justify-content-between align-items-center mb-3">
                    <h5 className="fw-bold mb-0 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                       <MoreHorizontal size={20} className="text-primary-custom" />
                       Meta Fields
                    </h5>
                    <button type="button" onClick={addMetaField} className="btn btn-sm btn-outline-primary fw-bold" style={{ borderColor: 'var(--border-color)', color: 'var(--text-color)' }}>
                       <Plus size={16} /> Add Meta
                    </button>
                  </div>
                  
                   <div className="d-flex flex-column gap-2">
                    {metaFields.map((field, index) => (
                       <div key={index} className="row g-2 align-items-center">
                           <div className="col-5">
                               <label className="small fw-bold text-muted-custom mb-1">Key</label>
                               <input 
                                 type="text" 
                                 className="form-control form-control-sm" 
                                 value={field.key} 
                                 onChange={(e) => handleMetaChange(index, 'key', e.target.value)} 
                                 style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                                 placeholder="e.g., vendor_type"
                               />
                           </div>
                            <div className="col-6">
                               <label className="small fw-bold text-muted-custom mb-1">Value</label>
                               <input 
                                 type="text" 
                                 className="form-control form-control-sm" 
                                 value={field.value} 
                                 onChange={(e) => handleMetaChange(index, 'value', e.target.value)} 
                                 style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                                 placeholder="e.g., food"
                               />
                           </div>
                           <div className="col-1 d-flex align-items-end justify-content-center pb-1">
                               <button type="button" onClick={() => removeMetaField(index)} className="btn btn-sm text-danger p-0 border-0"><X size={18} /></button>
                           </div>
                       </div>
                    ))}
                  </div>
                </div>

              </div>

              {/* RIGHT COLUMN: Sidebar */}
              <div className="col-12 col-lg-4">
                 
                 {/* Category Image */}
                 <div className="card-custom p-3 mb-3">
                   <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                     <ImageIcon size={20} className="text-primary-custom" />
                     Category Image
                   </h5>
                   <div className="d-flex align-items-center justify-content-center p-3 rounded-3 border-2 border-dashed transition-all hover-bg-custom position-relative overflow-hidden" 
                      style={{ borderColor: 'var(--border-color)', backgroundColor: 'var(--bg-color)', cursor: 'pointer', minHeight: '180px' }}>
                      <input type="file" accept="image/*" onChange={handleImageChange} className="position-absolute w-100 h-100 opacity-0 cursor-pointer" style={{ zIndex: 10 }} />
                      {imagePreview ? (
                          <div className="w-100 h-100 position-absolute top-0 start-0">
                              <img src={imagePreview} alt="Preview" className="w-100 h-100 object-fit-cover" />
                              <div className="position-absolute bottom-0 start-0 w-100 p-2 bg-dark bg-opacity-50 text-white text-center small">
                                  Click to change
                              </div>
                          </div>
                      ) : (
                        <div className="text-center">
                            <div className="mb-3 d-inline-flex p-3 rounded-circle shadow-sm" style={{ backgroundColor: 'var(--hover-bg)' }}>
                                <Upload size={24} className="text-primary-custom" />
                            </div>
                            <p className="small fw-bold mb-1" style={{ color: 'var(--text-color)' }}>Drop image here</p>
                            <p className="small text-muted-custom mb-0">or click to upload</p>
                        </div>
                      )}
                   </div>
                 </div>

                 {/* Banner Image */}
                 <div className="card-custom p-3 mb-3">
                   <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                     <ImageIcon size={20} className="text-primary-custom" />
                     Banner Image
                   </h5>
                   <div className="d-flex align-items-center justify-content-center p-3 rounded-3 border-2 border-dashed transition-all hover-bg-custom position-relative overflow-hidden" 
                      style={{ borderColor: 'var(--border-color)', backgroundColor: 'var(--bg-color)', cursor: 'pointer', minHeight: '180px' }}>
                      <input type="file" accept="image/*" onChange={handleBannerChange} className="position-absolute w-100 h-100 opacity-0 cursor-pointer" style={{ zIndex: 10 }} />
                      {bannerPreview ? (
                          <div className="w-100 h-100 position-absolute top-0 start-0">
                              <img src={bannerPreview} alt="Preview" className="w-100 h-100 object-fit-cover" />
                              <div className="position-absolute bottom-0 start-0 w-100 p-2 bg-dark bg-opacity-50 text-white text-center small">
                                  Click to change
                              </div>
                          </div>
                      ) : (
                        <div className="text-center">
                            <div className="mb-3 d-inline-flex p-3 rounded-circle shadow-sm" style={{ backgroundColor: 'var(--hover-bg)' }}>
                                <Upload size={24} className="text-primary-custom" />
                            </div>
                            <p className="small fw-bold mb-1" style={{ color: 'var(--text-color)' }}>Drop banner here</p>
                            <p className="small text-muted-custom mb-0">or click to upload</p>
                        </div>
                      )}
                   </div>
                 </div>

                 {/* Organization */}
                 <div className="card-custom p-3 mb-3">
                    <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                         <Layers size={20} className="text-primary-custom" />
                         Organization
                    </h5>
                    
                    <div className="mb-3">
                        <label className="form-label small fw-bold text-muted-custom">Scope</label>
                        <select 
                          name="scope" 
                          value={formData.scope} 
                          onChange={handleChange} 
                          className="form-select" 
                          style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                        >
                          <option value="home">Home</option>
                          <option value="ecommerce">E-commerce</option>
                        </select>
                    </div>
                    
                    <div className="mb-0">
                        <label className="form-label small fw-bold text-muted-custom">Parent Category</label>
                        <select 
                          name="parent_id" 
                          value={formData.parent_id} 
                          onChange={handleChange} 
                          className="form-select" 
                          style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                        >
                          <option value="">None (Top Level)</option>
                          {categories.map(cat => (
                            <option key={cat.id} value={cat.id}>{cat.title || cat.name}</option>
                          ))}
                        </select>
                        <small className="text-muted d-block mt-1" style={{ fontSize: '0.7rem' }}>Leave empty for top-level category</small>
                    </div>
                 </div>

                {/* Actions Footer */}
                <div className="col-12">
                   <div className="d-flex gap-2">
                       <button 
                         type="submit" 
                         disabled={loading} 
                         className="btn btn-success fw-bold flex-grow-1 shadow-sm py-2" 
                         style={{ background: 'linear-gradient(135deg, rgb(249, 115, 22) 0%, rgb(234, 88, 12) 100%)', border: 'none' }}
                       >
                         {loading ? "CREATING..." : "CREATE CATEGORY"}
                       </button>
                       <Link href="/admin/categories" className="btn btn-light fw-bold px-3 border" style={{ color: 'var(--text-color)' }}>
                         Cancel
                       </Link>
                   </div>
                </div>

               </div>

            </div>
          </form>
        </div>
      </div>
    </div>
  );
}
