"use client";

import PageHeader from "@/components/admin/PageHeader";
import ActionMenu from "@/components/admin/ActionMenu";
import DeleteModal from "@/components/admin/DeleteModal";
import { Search, Plus, Loader2, Image as ImageIcon, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight } from "lucide-react";
import { useState, useEffect } from "react";
import { BannerService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";

export default function AdminBannersPage() {
  const router = useRouter();
  const [banners, setBanners] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [filter, setFilter] = useState("all");
  const [deleteModalOpen, setDeleteModalOpen] = useState(false);
  const [selectedBanner, setSelectedBanner] = useState<any>(null);
  const [page, setPage] = useState(1);
  const [pagination, setPagination] = useState<any>(null);

  useEffect(() => {
    fetchBanners();
  }, [page, searchTerm, filter]);

  const fetchBanners = async () => {
    setLoading(true);
    try {
      const params: any = { page };
      if (searchTerm) {
        params.title = searchTerm;
      }
      if (filter !== "all") {
        params.scope = filter;
      }
      const response = await BannerService.getAll(params);
      const resData = response.data;
      setBanners(resData.data || []);
      
      setPagination(resData.meta || resData);
    } catch (error) {
      console.error("Failed to fetch banners:", error);
      toast.error("Failed to load banners");
    } finally {
      setLoading(false);
    }
  };

  const handleDeleteClick = (banner: any) => {
    setSelectedBanner(banner);
    setDeleteModalOpen(true);
  };

  const confirmDelete = async () => {
    if (!selectedBanner) return;
    try {
      await BannerService.delete(selectedBanner.id);
      toast.success("Banner deleted successfully");
      setDeleteModalOpen(false);
      setSelectedBanner(null);
      fetchBanners();
    } catch (error) {
      console.error("Failed to delete banner:", error);
      toast.error("Failed to delete banner");
    }
  };
  const getLocalizedValue = (value: any) => {
    if (!value) return "—";
    if (typeof value === 'string') {
      try {
        const parsed = JSON.parse(value);
        if (typeof parsed === 'object') return parsed.en || parsed.ar || Object.values(parsed)[0] || "—";
      } catch (e) {
        return value;
      }
    }
    if (typeof value === 'object') {
      return value.en || value.ar || Object.values(value)[0] || "—";
    }
    return String(value);
  };

  return (
    <div className="py-2">
      <PageHeader 
        title="Banner Management" 
        description="Manage promotional banners, sliders, and advertisements."
        actionLabel="Add New Banner"
        actionLink="/admin/banners/create"
      />

      <div className="card-custom overflow-hidden">
        {/* Toolbar */}
        <div className="p-4 border-bottom d-flex flex-column flex-md-row gap-3 justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
          <div className="d-flex p-1 rounded-3" style={{ backgroundColor: 'var(--bg-color)', border: '1px solid var(--border-color)' }}>
            {[
              { id: "all", label: "All Banners" },
              { id: "ecommerce", label: "eCommerce" },
            ].map((tab) => (
              <button
                key={tab.id}
                onClick={() => { setFilter(tab.id); setPage(1); }}
                className={`px-3 py-1.5 rounded-2 text-capitalize fw-bold border-0 transition-all ${filter === tab.id ? 'shadow-sm' : ''}`}
                style={{
                  fontSize: '0.85rem',
                  backgroundColor: filter === tab.id ? 'var(--text-color)' : 'transparent',
                  color: filter === tab.id ? 'var(--bg-color)' : 'var(--muted-text)',
                }}
              >
                {tab.label}
              </button>
            ))}
          </div>

          <div className="position-relative" style={{ width: '280px' }}>
            <Search className="position-absolute top-50 translate-middle-y ms-3" size={16} style={{ color: 'var(--muted-text)' }} />
            <input 
              type="text" 
              placeholder="Search by title..." 
              value={searchTerm}
              onChange={(e) => { setSearchTerm(e.target.value); setPage(1); }}
              className="form-control ps-5 py-2"
              style={{ 
                borderRadius: '12px',
                backgroundColor: 'var(--bg-color)',
                borderColor: 'var(--border-color)',
                color: 'var(--text-color)',
                fontSize: '0.9rem'
              }}
            />
          </div>
        </div>

        {/* Table */}
        <div className="table-responsive">
          {loading ? (
             <div className="d-flex flex-column align-items-center justify-content-center py-5 w-100">
                <Loader2 className="spin-animation text-primary mb-2" size={32} />
                <p className="text-muted-custom small">Loading banners...</p>
             </div>
          ) : banners.length === 0 ? (
            <div className="d-flex flex-column align-items-center justify-content-center py-5 opacity-50 w-100">
               <ImageIcon size={48} className="mb-2" />
               <p>No banners found.</p>
            </div>
          ) : (
            <>
              <table className="table mb-0 align-middle">
                <thead style={{ backgroundColor: 'var(--hover-bg)' }}>
                  <tr>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Banner</th>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Scope</th>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Order</th>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Created</th>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom text-end" style={{ borderColor: 'var(--border-color)' }}>Actions</th>
                  </tr>
                </thead>
                <tbody>
                  {banners.map((banner) => (
                    <tr key={banner.id} style={{ borderBottom: '1px solid var(--border-color)' }}>
                      <td className="px-4 py-3">
                        <div className="d-flex align-items-center gap-3">
                          <div className="rounded-3 border overflow-hidden bg-light d-flex align-items-center justify-content-center" style={{ width: '70px', height: '40px' }}>
                            {banner.mediaurls?.images?.[0]?.default ? (
                              <img src={banner.mediaurls.images[0].default} alt={getLocalizedValue(banner.title)} className="w-100 h-100 object-fit-cover" />
                            ) : (
                              <ImageIcon size={16} className="text-muted" />
                            )}
                          </div>
                          <div>
                            <p className="fw-bold mb-0" style={{ color: 'var(--text-color)' }}>{getLocalizedValue(banner.title)}</p>
                            <p className="extra-small text-muted-custom mb-0">ID: #{banner.id}</p>
                          </div>
                        </div>
                      </td>
                      <td className="px-4 py-3">
                        <span className="badge rounded-pill fw-bold border" style={{
                           backgroundColor: 'rgba(59, 130, 246, 0.1)',
                           color: '#3b82f6',
                           borderColor: 'rgba(59, 130, 246, 0.2)',
                           padding: '0.4em 0.8em'
                        }}>
                          {banner.meta?.scope || "System"}
                        </span>
                      </td>
                      <td className="px-4 py-3">
                        <span className="small fw-bold">{banner.sort_order || 0}</span>
                      </td>
                      <td className="px-4 py-3 small text-muted-custom font-medium">
                        {new Date(banner.created_at || Date.now()).toLocaleDateString()}
                      </td>
                      <td className="px-4 py-3 text-end">
                        <ActionMenu 
                          onView={() => router.push(`/admin/banners/${banner.id}`)}
                          onEdit={() => router.push(`/admin/banners/${banner.id}/edit`)}
                          onDelete={() => handleDeleteClick(banner)}
                        />
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>

              {/* Pagination */}
              {pagination && pagination.last_page > 1 && (
                <div className="p-4 border-top d-flex justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
                  <div className="small text-muted-custom">
                    Showing <span className="fw-bold text-dark">{pagination.from || 0}</span> to <span className="fw-bold text-dark">{pagination.to || 0}</span> of <span className="fw-bold text-dark">{pagination.total || 0}</span> results
                  </div>
                  <div className="d-flex gap-2 flex-wrap">
                    {/* First Page Button */}
                    <button
                      onClick={() => setPage(1)}
                      disabled={pagination.current_page === 1}
                      className="btn btn-white border btn-sm d-flex align-items-center justify-content-center p-2 rounded-3 shadow-sm transition-all"
                      title="First Page"
                    >
                      <ChevronsLeft size={16} />
                    </button>
                    
                    {/* Previous Button */}
                    <button
                      onClick={() => setPage(prev => Math.max(prev - 1, 1))}
                      disabled={pagination.current_page === 1}
                      className="btn btn-white border btn-sm d-flex align-items-center justify-content-center p-2 rounded-3 shadow-sm transition-all"
                      title="Previous Page"
                    >
                      <ChevronLeft size={16} />
                    </button>
                    
                    {/* Page Numbers */}
                    <div className="d-flex align-items-center gap-1 flex-wrap">
                      {pagination.links
                        ?.filter((link: any) => {
                          const pageNum = parseInt(link.label);
                          return !isNaN(pageNum) && link.label.trim() === pageNum.toString();
                        })
                        .map((link: any, index: number) => {
                          const isCurrent = link.active;
                          const pageNumber = parseInt(link.label);
                          
                          return (
                            <button
                              key={index}
                              onClick={() => setPage(pageNumber)}
                              disabled={isCurrent}
                              className={`btn btn-sm ${isCurrent ? 'btn-primary' : 'btn-light border'}`}
                              style={{ minWidth: '32px', height: '32px', padding: '0 0.5rem' }}
                            >
                              {link.label}
                            </button>
                          );
                        })}
                    </div>
                    
                    {/* Next Button */}
                    <button
                      onClick={() => setPage(prev => Math.min(prev + 1, pagination.last_page))}
                      disabled={pagination.current_page === pagination.last_page}
                      className="btn btn-white border btn-sm d-flex align-items-center justify-content-center p-2 rounded-3 shadow-sm transition-all"
                      title="Next Page"
                    >
                      <ChevronRight size={16} />
                    </button>
                    
                    {/* Last Page Button */}
                    <button
                      onClick={() => setPage(pagination.last_page)}
                      disabled={pagination.current_page === pagination.last_page}
                      className="btn btn-white border btn-sm d-flex align-items-center justify-content-center p-2 rounded-3 shadow-sm transition-all"
                      title="Last Page"
                    >
                      <ChevronsRight size={16} />
                    </button>
                  </div>
                </div>
              )}
            </>
          )}
        </div>
      </div>

      <DeleteModal 
        isOpen={deleteModalOpen}
        onClose={() => setDeleteModalOpen(false)}
        onConfirm={confirmDelete}
        itemName={getLocalizedValue(selectedBanner?.title) || undefined}
        title="Delete Banner"
        description="Are you sure you want to delete this banner? This action cannot be undone."
      />
    </div>
  );
}
