"use client";

import PageHeader from "@/components/admin/PageHeader";
import { Save, X, Image as ImageIcon, Plus, Trash2, Loader2, ChevronLeft, Globe, Hash, Settings } from "lucide-react";
import Link from "next/link";
import { useState } from "react";
import { BannerService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";

export default function CreateBannerPage() {
  const router = useRouter();
  const [loading, setLoading] = useState(false);
  const [image, setImage] = useState<File | null>(null);
  const [imagePreview, setImagePreview] = useState<string | null>(null);

  // Form State
  const [titles, setTitles] = useState({
    en: "",
    ar: ""
  });
  const [sortOrder, setSortOrder] = useState("1");
  const [metaFields, setMetaFields] = useState<any[]>([
    { key: "scope", value: "ecommerce" }
  ]);

  const handleImageChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      setImage(file);
      const reader = new FileReader();
      reader.onloadend = () => {
        setImagePreview(reader.result as string);
      };
      reader.readAsDataURL(file);
    }
  };

  const addMetaField = () => {
    setMetaFields([...metaFields, { key: "", value: "" }]);
  };

  const removeMetaField = (index: number) => {
    setMetaFields(metaFields.filter((_, i) => i !== index));
  };

  const updateMetaField = (index: number, field: "key" | "value", newValue: string) => {
    const updated = [...metaFields];
    updated[index][field] = newValue;
    setMetaFields(updated);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      const formData = new FormData();
      
      // Title Translations mapping
      const translations = [
        { language: "en", text: titles.en },
        { language: "ar", text: titles.ar }
      ];
      
      formData.append("title_translations", JSON.stringify(translations));

      // PHP side often needs _title for the main record title attribute
      formData.append("_title", titles.en);

      formData.append("sort_order", sortOrder);

      // Meta fields to JSON
      const metaObj: any = {};
      metaFields.forEach(field => {
        if (field.key) metaObj[field.key] = field.value;
      });
      formData.append("meta", JSON.stringify(metaObj));

      if (image) {
        formData.append("image", image);
      }

      await BannerService.create(formData);
      toast.success("Banner created successfully!");
      router.push("/admin/banners");
    } catch (err: any) {
      console.error(err);
      toast.error(err.response?.data?.message || "Failed to create banner. Check console for details.");
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="py-2">
      <div className="d-flex align-items-center gap-3 mb-4">
        <Link href="/admin/banners" className="btn btn-light rounded-circle p-2 border shadow-sm">
          <ChevronLeft size={20} />
        </Link>
        <div>
          <h2 className="fw-bold mb-1" style={{ color: 'var(--text-color)' }}>Create New Banner</h2>
          <p className="lead fs-6 mb-0" style={{ color: 'var(--muted-text)' }}>Add a promotional banner to the system.</p>
        </div>
      </div>

      <form onSubmit={handleSubmit}>
        <div className="row g-4">
          
          {/* LEFT COLUMN: Titles & Details */}
          <div className="col-12 col-lg-8">
            <div className="card-custom p-4 mb-4">
              <h5 className="fw-bold mb-4 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                <Globe size={20} className="text-primary-custom" />
                Title Translations
              </h5>
              
              <div className="row g-3">
                <div className="col-12">
                  <label className="form-label small fw-bold text-muted-custom">Title (English)</label>
                  <input 
                    type="text" 
                    value={titles.en} 
                    onChange={(e) => setTitles({...titles, en: e.target.value})} 
                    className="form-control" 
                    placeholder="Summer Sale 2024" 
                    required 
                  />
                </div>
                <div className="col-12">
                  <label className="form-label small fw-bold text-muted-custom">Title (Arabic)</label>
                  <input 
                    type="text" 
                    value={titles.ar} 
                    onChange={(e) => setTitles({...titles, ar: e.target.value})} 
                    className="form-control text-end" 
                    dir="rtl"
                    placeholder="تنزيلات الصيف ٢٠٢٤" 
                    required 
                  />
                </div>
              </div>
            </div>

            <div className="card-custom p-4">
              <div className="d-flex justify-content-between align-items-center mb-4">
                <h5 className="fw-bold mb-0 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                  <Settings size={20} className="text-primary-custom" />
                  Meta Fields (Configuration)
                </h5>
                <button 
                  type="button" 
                  onClick={addMetaField}
                  className="btn btn-primary-custom btn-sm px-3 rounded-3 d-flex align-items-center gap-2"
                >
                  <Plus size={16} />
                  Add Meta Field
                </button>
              </div>

              <div className="meta-fields-container">
                {metaFields.map((field, index) => (
                  <div key={index} className="row g-2 mb-3 align-items-end">
                    <div className="col-5">
                      <label className="form-label small fw-bold text-muted-custom">Key</label>
                      <input 
                        type="text" 
                        value={field.key} 
                        onChange={(e) => updateMetaField(index, "key", e.target.value)} 
                        className="form-control form-control-sm" 
                        placeholder="scope" 
                      />
                    </div>
                    <div className="col-5">
                      <label className="form-label small fw-bold text-muted-custom">Value</label>
                      <input 
                        type="text" 
                        value={field.value} 
                        onChange={(e) => updateMetaField(index, "value", e.target.value)} 
                        className="form-control form-control-sm" 
                        placeholder="ecommerce" 
                      />
                    </div>
                    <div className="col-2">
                      <button 
                        type="button" 
                        onClick={() => removeMetaField(index)}
                        className="btn btn-light-danger btn-sm p-2 rounded-3"
                        disabled={metaFields.length === 1 && field.key === 'scope'}
                      >
                        <Trash2 size={16} />
                      </button>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>

          {/* RIGHT COLUMN: Image & Sorting */}
          <div className="col-12 col-lg-4">
            <div className="card-custom p-4 mb-4">
              <h5 className="fw-bold mb-4 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                <ImageIcon size={20} className="text-primary-custom" />
                Banner Media
              </h5>
              
              <div className="image-upload-area rounded-4 border-2 border-dashed d-flex flex-column align-items-center justify-content-center p-4 text-center transition-all hover:bg-light" style={{ borderColor: 'var(--border-color)', minHeight: '200px', cursor: 'pointer', position: 'relative' }} onClick={() => document.getElementById('banner-image')?.click()}>
                {imagePreview ? (
                  <img src={imagePreview} alt="Preview" className="img-fluid rounded-3 shadow-sm max-h-40" style={{ maxHeight: '200px', width: 'auto' }} />
                ) : (
                  <>
                    <div className="p-3 bg-primary bg-opacity-10 rounded-circle mb-3">
                      <ImageIcon size={32} className="text-primary" />
                    </div>
                    <p className="small fw-bold mb-1">Click to upload banner</p>
                    <p className="extra-small text-muted mb-0">Recommended size: 1200x400px</p>
                  </>
                )}
                <input 
                  id="banner-image" 
                  type="file" 
                  className="d-none" 
                  accept="image/*" 
                  onChange={handleImageChange} 
                />
              </div>
            </div>

            <div className="card-custom p-4 mb-4">
              <h5 className="fw-bold mb-4 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                <Hash size={20} className="text-primary-custom" />
                Sorting
              </h5>
              <div className="mb-3">
                <label className="form-label small fw-bold text-muted-custom">Sort Order</label>
                <input 
                  type="number" 
                  value={sortOrder} 
                  onChange={(e) => setSortOrder(e.target.value)} 
                  className="form-control" 
                  placeholder="1" 
                />
                <p className="extra-small text-muted mt-2 mb-0">Lower numbers appear first in the app slider.</p>
              </div>
            </div>

            <div className="d-flex flex-column gap-2">
              <button 
                type="submit" 
                disabled={loading} 
                className="btn btn-primary-custom fw-bold py-3 shadow-sm d-flex align-items-center justify-content-center gap-2 border-0 text-white"
                style={{ background: 'linear-gradient(135deg, #f97316 0%, #ea580c 100%)' }}
              >
                {loading ? <Loader2 size={18} className="spin-animation" /> : <Save size={18} />}
                SAVE BANNER
              </button>
              <Link href="/admin/banners" className="btn btn-light fw-bold py-2 border shadow-sm">
                Cancel
              </Link>
            </div>
          </div>
        </div>
      </form>
    </div>
  );
}
